<?xml version="1.0" encoding="ISO-8859-1"?>

<!--
     Stylesheet for IEEE sources.
     
     Version  : 1.0 - Initial release.
     
                2.0 - Rewrite of most of the bibliography presentation code.
                
                2.1 - Added possibility to combine several in-text citations
                      using the \m tag in the field code. For example:
                        {CITATION Tag1 \m Tag2 \m Tag3}
                        
                      Moved all in-text citation functionality into one function
                      rather than having a separate function for every b:SourceType.
     
                2.2 - Fixed some bugs when abbreviating first and middle names 
                      when they consist out of more than two parts.
                      (Remark: this bugs also exists in the styles provided by
                      Microsoft (e.g.: APA.xsl))
                      
                2.3 - Made in-text citations clickable. To enable, the value
                      for the 'InTextCitationsAsLink'-variable should be set
                      to 'true'.
                
                2.4 - Fixed a double dot-bug when an edition was displayed and 
                      no editors where displayed.
                      Fixed a bug in the important field definition of 
                      miscellaneous entries.
                
                2.5 - Added MsoBibliography style support. Now the bibliography
                      is automatically formatted according to the 'Bibliography'
                      style defined in Word.
     
     Author(s): Yves Dhondt (yves.dhondt@gmail.com)
     
     Copyright: Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated 
                documentation files (the "Software"), to deal in the Software without restriction, including without limitation 
                the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, 
                and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

                The above copyright notice and this permission notice shall be included in all copies or substantial portions 
                of the Software.

                THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED 
                TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL 
                THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF 
                CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
                IN THE SOFTWARE.
     
     Usage    : Copy the stylesheet to the 'Style'-directory of Office 2007. On most 
                computers this will be somewhere like
                    %programfiles%\Microsoft Office\Office12\Bibliography\Style 
                The style will show up in the 'Reference' bar next time you launch 
                Word 2007.
     
     Sources  : Formatting is based on three different documents:
                1)IEEE style manual
                       http://www.ieee.org/portal/cms_docs_iportals/iportals/publications/authors/transjnl/stylemanual.pdf
                2)IEEE standards style manual 
                       http://standards.ieee.org/guides/style/2007_Style_Manual.pdf
                3)IEEEtran.bst stylesheet for LaTeX
                       http://www.ctan.org/get/macros/latex/contrib/IEEEtran/bibtex/IEEEtran.bst
                
                In general, the style which displays most fields of a source is 
                used in this stylesheet. The idea is that readers of an article 
                should get as much information as possible.

     Remarks  : The following types are currently supported:
                    Book
                    BookSection
                    JournalArticle
                    ArticleInAPeriodical
                    ConferenceProceedings
                    Report (use Report with type containing 'rep')
                    InternetSite (alternatively, use ElectronicSource)
                    DocumentFromInternetSite (alternatively, use ElectronicSource)
                    Patent
                    Standard (use Report)
                    M.S. thesis (use Report with type containing 'm' and 'thesis')
                    PhD. dissertation (use Report with type containing 'phd')
                    Blog (use ElectronicSource with medium being 'Blog')
                    CD-ROM (use ElectronicSource with medium containing 'CD-rom')
                    
                Other types will be implemented on a on-demand basis.    
-->

<xsl:stylesheet version="1.0"
        xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
        xmlns:msxsl="urn:schemas-microsoft-com:xslt"
        xmlns:b="http://schemas.openxmlformats.org/officeDocument/2006/bibliography">

  <!-- +++++++++++++++++++++++++++ -->
  <!-- Begin of parameter section. -->
  <!-- +++++++++++++++++++++++++++ -->

  <!-- 
       Variables in this section define common behaviour for multiple types.
       If you wish to have different behaviour for different types, you should
       replace the variables at the types themselves with the required values.
   -->

  <!-- Indicates if in-text citations should be links to the bibliography or not. -->
  <!-- Valid values are 'true' and 'false'. -->
  <xsl:variable name="InTextCitationAsLink" select="'true'" />
  
  <!-- Indicates if URLs should be displayed as hyperlinks or not. -->
  <!-- Valid values are 'true' and 'false'. -->
  <xsl:variable name="DisplayURLAsLink" select="'true'" />

  <!-- Indicates the maximum number of authors before the formatting should
       fall back to one author followed by 'et al.'. -->
  <xsl:variable name="MaxNumberOfAuthors" select="5" />

  <!-- Indicates the maximum number of editors before the formatting should
       fall back to one editor followed by 'et al.'. -->
  <xsl:variable name="MaxNumberOfEditors" select="3" />

  <!-- +++++++++++++++++++++++++ -->
  <!-- End of parameter section. -->
  <!-- +++++++++++++++++++++++++ -->


  <!-- Variables used to convert from uppercase to lowercase and vice versa. -->
  <xsl:variable name="lower">abcdefghijklmnopqrstuvwxyz</xsl:variable>
  <xsl:variable name="upper">ABCDEFGHIJKLMNOPQRSTUVWXYZ</xsl:variable>

  <xsl:output method="html" encoding="us-ascii"/>

  <!-- Handling of basic requests -->
  <xsl:template match="/">
    <xsl:choose>

      <!-- Short name for the style to use. -->
      <!-- This is the name that will -->
      <xsl:when test="b:StyleName">
        <xsl:text>IEEE</xsl:text>
      </xsl:when>

      <!-- Version information. -->
      <!-- Version information should be in the form <xsl:text>yyyy.m.dd</xsl:text>. -->
      <xsl:when test="b:Version">
        <xsl:text>2008.7.16</xsl:text>
      </xsl:when>

      <!-- Retrieve the most important fields for a certain type of citation. -->
      <xsl:when test="b:GetImportantFields">
        <xsl:call-template name="SelectImportantFields" />
      </xsl:when>

      <!-- Formats a citation for display. -->
      <xsl:when test="b:Citation">
        <html xmlns="http://www.w3.org/TR/REC-html40">
          <head>
          </head>
          <body>
            <p>
              <!-- Opening in-text citation. -->
              <xsl:if test="/b:Citation/b:FirstAuthor">
                <xsl:text>[</xsl:text>
              </xsl:if>

              <!-- Display in-text citation. -->
              <xsl:choose>
                <!-- Display in-text citation as link. -->
                <xsl:when test="$InTextCitationAsLink = 'true'">
                  <a href="#{/b:Citation/b:Source/b:Tag}">
                    <xsl:value-of select="/b:Citation/b:Source/b:RefOrder"/>
                  </a>
                </xsl:when>
                <!-- Display in-text citation without link. -->
                <xsl:otherwise>
                  <xsl:value-of select="/b:Citation/b:Source/b:RefOrder"/>
                </xsl:otherwise>
              </xsl:choose>
                            
              <xsl:choose>
                <!-- Closing in-text citation. -->
                <xsl:when test="/b:Citation/b:LastAuthor">
                  <xsl:text>]</xsl:text>
                </xsl:when>
                <!-- More citations to follow. -->
                <xsl:otherwise>
                  <xsl:text>,</xsl:text>
                </xsl:otherwise>
              </xsl:choose>
            </p>
          </body>
        </html>

      </xsl:when>

      <!-- Formats a bibliography for display. -->
      <xsl:when test="b:Bibliography">
        <html xmlns="http://www.w3.org/TR/REC-html40">
          <head>
            <style>
              p.MsoBibliography
            </style>
          </head>
          <body>
            <table width="100%">
              <xsl:for-each select="/b:Bibliography/b:Source">
                <xsl:sort select="b:RefOrder" data-type="number"/>
                <tr>
                  <td valign="top">
                    <p class="MsoBibliography">
                      <xsl:text>[</xsl:text>
                      <xsl:choose>
                        <!-- Display in-text citation as link. -->
                        <xsl:when test="$InTextCitationAsLink = 'true'">
                          <a name="{./b:Tag}">
                            <xsl:value-of select="./b:RefOrder"/>
                          </a>
                        </xsl:when>
                        <!-- Display in-text citation without link. -->
                        <xsl:otherwise>
                          <xsl:value-of select="./b:RefOrder"/>
                        </xsl:otherwise>
                      </xsl:choose>
                      <xsl:text>]</xsl:text>
                    </p>
                  </td>
                  <td valign="top">
                    <p class="MsoBibliography">
                      <xsl:call-template name="FormatReference">
                        <xsl:with-param name ="reference" select="." />
                      </xsl:call-template>
                    </p>
                  </td>
                </tr>
              </xsl:for-each>
            </table>
          </body>
        </html>
      </xsl:when>
    </xsl:choose>
  </xsl:template>

  <!-- Begin of picking specific functions for different types of sources. -->

  <!-- Template for selecting the important fields of a certain reference type. -->
  <xsl:template name="SelectImportantFields">
    <xsl:choose>
      <xsl:when test="b:GetImportantFields/b:SourceType='Book'">
        <xsl:call-template name="SelectImportantFieldsForBook" />
      </xsl:when>
      <xsl:when test="b:GetImportantFields/b:SourceType='BookSection'">
        <xsl:call-template name="SelectImportantFieldsForBookSection" />
      </xsl:when>
      <xsl:when test="b:GetImportantFields/b:SourceType='JournalArticle'">
        <xsl:call-template name="SelectImportantFieldsForJournalArticle" />
      </xsl:when>
      <xsl:when test="b:GetImportantFields/b:SourceType='ArticleInAPeriodical'">
        <xsl:call-template name="SelectImportantFieldsForArticleInAPeriodical" />
      </xsl:when>
      <xsl:when test="b:GetImportantFields/b:SourceType='ConferenceProceedings'">
        <xsl:call-template name="SelectImportantFieldsForConferenceProceedings" />
      </xsl:when>
      <xsl:when test="b:GetImportantFields/b:SourceType='Report'">
        <xsl:call-template name="SelectImportantFieldsForReport" />
      </xsl:when>
      <xsl:when test="b:GetImportantFields/b:SourceType='InternetSite'">
        <xsl:call-template name="SelectImportantFieldsForInternetSite" />
      </xsl:when>
      <xsl:when test="b:GetImportantFields/b:SourceType='DocumentFromInternetSite'">
        <xsl:call-template name="SelectImportantFieldsForDocumentFromInternetSite" />
      </xsl:when>
      <xsl:when test="b:GetImportantFields/b:SourceType='ElectronicSource'">
        <xsl:call-template name="SelectImportantFieldsForElectronicSource" />
      </xsl:when>
      <xsl:when test="b:GetImportantFields/b:SourceType='Art'">
        <xsl:call-template name="SelectImportantFieldsForArt" />
      </xsl:when>
      <xsl:when test="b:GetImportantFields/b:SourceType='SoundRecording'">
        <xsl:call-template name="SelectImportantFieldsForSoundRecording" />
      </xsl:when>
      <xsl:when test="b:GetImportantFields/b:SourceType='Performance'">
        <xsl:call-template name="SelectImportantFieldsForPerformance" />
      </xsl:when>
      <xsl:when test="b:GetImportantFields/b:SourceType='Film'">
        <xsl:call-template name="SelectImportantFieldsForFilm" />
      </xsl:when>
      <xsl:when test="b:GetImportantFields/b:SourceType='Interview'">
        <xsl:call-template name="SelectImportantFieldsForInterview" />
      </xsl:when>
      <xsl:when test="b:GetImportantFields/b:SourceType='Patent'">
        <xsl:call-template name="SelectImportantFieldsForPatent" />
      </xsl:when>
      <xsl:when test="b:GetImportantFields/b:SourceType='Case'">
        <xsl:call-template name="SelectImportantFieldsForCase" />
      </xsl:when>
      <xsl:when test="b:GetImportantFields/b:SourceType='Misc'">
        <xsl:call-template name="SelectImportantFieldsForMisc" />
      </xsl:when>
    </xsl:choose>

  </xsl:template>

  <!-- Template for selecting a reference type to format the reference correctly. -->
  <!--   As parameter, a single entry from the bibliography is required. -->
  <xsl:template name="FormatReference">
    <xsl:param name="reference" />

    <xsl:choose>
      <xsl:when test="$reference/b:SourceType='Book'">
        <xsl:call-template name="FormatBookReference">
          <xsl:with-param name="reference" select="$reference" />
        </xsl:call-template>
      </xsl:when>
      <xsl:when test="$reference/b:SourceType='BookSection'">
        <xsl:call-template name="FormatBookSectionReference">
          <xsl:with-param name="reference" select="$reference" />
        </xsl:call-template>
      </xsl:when>
      <xsl:when test="$reference/b:SourceType='JournalArticle'">
        <xsl:call-template name="FormatJournalArticleReference">
          <xsl:with-param name="reference" select="$reference" />
        </xsl:call-template>
      </xsl:when>
      <xsl:when test="$reference/b:SourceType='ArticleInAPeriodical'">
        <xsl:call-template name="FormatArticleInAPeriodicalReference">
          <xsl:with-param name="reference" select="$reference" />
        </xsl:call-template>
      </xsl:when>
      <xsl:when test="$reference/b:SourceType='ConferenceProceedings'">
        <xsl:call-template name="FormatConferenceProceedingsReference">
          <xsl:with-param name="reference" select="$reference" />
        </xsl:call-template>
      </xsl:when>
      <xsl:when test="$reference/b:SourceType='Report'">
        <xsl:call-template name="FormatReportReference">
          <xsl:with-param name="reference" select="$reference" />
        </xsl:call-template>
      </xsl:when>
      <xsl:when test="$reference/b:SourceType='InternetSite'">
        <xsl:call-template name="FormatInternetSiteReference">
          <xsl:with-param name="reference" select="$reference" />
        </xsl:call-template>
      </xsl:when>
      <xsl:when test="$reference/b:SourceType='DocumentFromInternetSite'">
        <xsl:call-template name="FormatDocumentFromInternetSiteReference">
          <xsl:with-param name="reference" select="$reference" />
        </xsl:call-template>
      </xsl:when>
      <xsl:when test="$reference/b:SourceType='ElectronicSource'">
        <xsl:call-template name="FormatElectronicSourceReference">
          <xsl:with-param name="reference" select="$reference" />
        </xsl:call-template>
      </xsl:when>
      <xsl:when test="$reference/b:SourceType='Art'">
        <xsl:call-template name="FormatArtReference">
          <xsl:with-param name="reference" select="$reference" />
        </xsl:call-template>
      </xsl:when>
      <xsl:when test="$reference/b:SourceType='SoundRecording'">
        <xsl:call-template name="FormatSoundRecordingReference">
          <xsl:with-param name="reference" select="$reference" />
        </xsl:call-template>
      </xsl:when>
      <xsl:when test="$reference/b:SourceType='Performance'">
        <xsl:call-template name="FormatPerformanceReference">
          <xsl:with-param name="reference" select="$reference" />
        </xsl:call-template>
      </xsl:when>
      <xsl:when test="$reference/b:SourceType='Film'">
        <xsl:call-template name="FormatFilmReference">
          <xsl:with-param name="reference" select="$reference" />
        </xsl:call-template>
      </xsl:when>
      <xsl:when test="$reference/b:SourceType='Interview'">
        <xsl:call-template name="FormatInterviewReference">
          <xsl:with-param name="reference" select="$reference" />
        </xsl:call-template>
      </xsl:when>
      <xsl:when test="$reference/b:SourceType='Patent'">
        <xsl:call-template name="FormatPatentReference">
          <xsl:with-param name="reference" select="$reference" />
        </xsl:call-template>
      </xsl:when>
      <xsl:when test="$reference/b:SourceType='Case'">
        <xsl:call-template name="FormatCaseReference">
          <xsl:with-param name="reference" select="$reference" />
        </xsl:call-template>
      </xsl:when>
      <xsl:when test="$reference/b:SourceType='Misc'">
        <xsl:call-template name="FormatMiscReference">
          <xsl:with-param name="reference" select="$reference" />
        </xsl:call-template>
      </xsl:when>
      <xsl:otherwise>
        <xsl:text>Unsupported reference type: </xsl:text>
        <xsl:value-of select="b:SourceType" />
      </xsl:otherwise>
    </xsl:choose>

  </xsl:template>

  <!-- End of picking specific functions for different types of sources. -->

  <!-- Begin of templates for Book. -->

  <xsl:template name="SelectImportantFieldsForBook">
    <b:ImportantFields>
      <b:ImportantField>
        <xsl:text>b:Author/b:Author/b:NameList</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Title</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Author/b:Editor/b:NameList</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:City</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:CountryRegion</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Publisher</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Year</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Edition</xsl:text>
      </b:ImportantField>
    </b:ImportantFields>
  </xsl:template>

  <xsl:template name="FormatBookReference">
    <xsl:param name="reference" />

    <!-- Format the authors, if not available, fall back to the editors. -->
    <xsl:variable name="authors">
      <xsl:choose>
        <xsl:when test="string-length($reference/b:Author/b:Author) = 0">
          <xsl:call-template name="FormatEditors">
            <xsl:with-param name="list" select="$reference/b:Author/b:Editor" />
            <xsl:with-param name="maxEditors" select="$MaxNumberOfEditors" />
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="FormatAuthors">
            <xsl:with-param name="list" select="$reference/b:Author/b:Author" />
            <xsl:with-param name="maxAuthors" select="$MaxNumberOfAuthors" />
          </xsl:call-template>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <!-- Format the title, if not available, fall back to the book title. -->
    <xsl:variable name="title">
      <i>
        <xsl:choose>
          <xsl:when test="string-length($reference/b:Title) > 0">
            <xsl:value-of select="$reference/b:Title"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="$reference/b:BookTitle"/>
          </xsl:otherwise>
        </xsl:choose>
      </i>
      <xsl:if test="string-length($reference/b:Title) > 0 or string-length($reference/b:BookTitle) > 0">
        <xsl:if test="string-length($reference/b:Edition) > 0">
          <xsl:text>, </xsl:text>
          <xsl:call-template name="FormatOrdinal">
            <xsl:with-param name="number" select="$reference/b:Edition" />
          </xsl:call-template>
          <xsl:text> ed.</xsl:text>
        </xsl:if>
      </xsl:if>
    </xsl:variable>

    <!-- Format the editors. -->
    <xsl:variable name="editors">
      <xsl:if test="string-length($reference/b:Author/b:Author) > 0 and string-length($reference/b:Author/b:Editor) > 0">
        <xsl:call-template name="FormatEditors">
          <xsl:with-param name="list" select="$reference/b:Author/b:Editor" />
          <xsl:with-param name="maxEditors" select="$MaxNumberOfEditors" />
        </xsl:call-template>
      </xsl:if>
    </xsl:variable>

    <!-- Format the location. -->
    <xsl:variable name="location">
      <xsl:if test="string-length($reference/b:City) >0">
        <xsl:value-of select="$reference/b:City" />
        <xsl:if test="string-length($reference/b:StateProvince) >0 or string-length($reference/b:CountryRegion) > 0">
          <xsl:text>, </xsl:text>
        </xsl:if>
      </xsl:if>

      <xsl:if test="string-length($reference/b:StateProvince) >0">
        <xsl:value-of select="$reference/b:StateProvince" />
        <xsl:if test="string-length($reference/b:CountryRegion) > 0">
          <xsl:text>, </xsl:text>
        </xsl:if>
      </xsl:if>

      <xsl:value-of select="$reference/b:CountryRegion" />
    </xsl:variable>

    <!-- Format publisher. -->
    <xsl:variable name="publisher">
      <xsl:value-of select="$reference/b:Publisher"/>
    </xsl:variable>

    <!-- Format date. -->
    <xsl:variable name="date">
      <xsl:call-template name="FormatDate">
        <xsl:with-param name="month" select="$reference/b:Month" />
        <xsl:with-param name="year" select="$reference/b:Year" />
      </xsl:call-template>
    </xsl:variable>

    <!-- Format the volume. -->
    <xsl:variable name="volume">
      <xsl:if test="string-length($reference/b:Volume) > 0">
        <xsl:text>vol. </xsl:text>
        <xsl:value-of select="$reference/b:Volume" />
      </xsl:if>
    </xsl:variable>

    <!-- Display the authors. -->
    <xsl:if test="string-length($authors) > 0">
      <xsl:value-of select="$authors"/>
      <xsl:if test="string-length($title) > 0 or string-length($editors) > 0">
        <xsl:text>, </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the title. -->
    <xsl:if test="string-length($title) > 0">
      <xsl:copy-of select="($title)"/>
      <xsl:if test="string-length($editors) > 0">
        <xsl:text>, </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the editors. -->
    <xsl:choose>
      <xsl:when test="string-length($editors) > 0">
        <xsl:value-of select="$editors"/>
        <xsl:if test="string-length($location) > 0 or string-length($publisher) > 0 or string-length($date) > 0 or string-length($volume) > 0">
          <xsl:text> </xsl:text>
        </xsl:if>
      </xsl:when>
      <xsl:otherwise>
        <xsl:if test="string-length($location) > 0 or string-length($publisher) > 0 or string-length($date) > 0 or string-length($volume) > 0">
          <xsl:if test="substring($title, string-length($title) , 1) != '.'">
            <xsl:text>.</xsl:text>
          </xsl:if>
          <xsl:text> </xsl:text>
        </xsl:if>
      </xsl:otherwise>
    </xsl:choose>

    <!-- Display location and publisher. -->
    <xsl:if test="string-length($location) > 0">
      <xsl:value-of select="$location" />
      <xsl:choose>
        <xsl:when test="string-length($publisher) > 0">
          <xsl:text>: </xsl:text>
        </xsl:when>
        <xsl:when test="string-length($date) > 0 or string-length($volume) > 0">
          <xsl:text>, </xsl:text>
        </xsl:when>
      </xsl:choose>
    </xsl:if>

    <!-- Display the publisher. -->
    <xsl:if test="string-length($publisher) > 0">
      <xsl:value-of select="$publisher"/>
      <xsl:if test="string-length($date) > 0 or string-length($volume) > 0">
        <xsl:text>, </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the date. -->
    <xsl:if test="string-length($date) > 0">
      <xsl:value-of select="$date"/>
      <xsl:if test="string-length($volume) > 0">
        <xsl:text>, </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the volume. -->
    <xsl:value-of select="$volume"/>

    <!-- Display comments. -->
    <xsl:if test="string-length($reference/b:Comments) > 0">
      <xsl:text>, </xsl:text>
      <xsl:value-of select="$reference/b:Comments"/>
    </xsl:if>

    <!-- Display final '.' -->
    <xsl:if test="string-length($authors) > 0 or string-length($title) > 0 or string-length($editors) > 0 or string-length($location) > 0 or string-length($publisher) > 0 or string-length($date) > 0 or string-length($volume) > 0">
      <xsl:text>.</xsl:text>
      <xsl:if test="string-length($reference/b:URL) > 0">
        <xsl:text> </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display online availability. -->
    <xsl:if test="string-length($reference/b:URL) > 0">
      <xsl:call-template name="FormatURL">
        <xsl:with-param name="url" select="$reference/b:URL" />
        <xsl:with-param name="displayAsLink" select="$DisplayURLAsLink" />
      </xsl:call-template>
    </xsl:if>

  </xsl:template>

  <!-- End of templates for Book. -->

  <!-- Begin of templates for BookSection. -->

  <xsl:template name="SelectImportantFieldsForBookSection">
    <b:ImportantFields>
      <b:ImportantField>
        <xsl:text>b:Author/b:Author/b:NameList</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Title</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Author/b:BookAuthor/b:NameList</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:BookTitle</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Author/b:Editor/b:NameList</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:City</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:CountryRegion</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Publisher</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Year</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:ChapterNumber</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Pages</xsl:text>
      </b:ImportantField>
    </b:ImportantFields>
  </xsl:template>

  <xsl:template name="FormatBookSectionReference">
    <xsl:param name="reference" />

    <xsl:choose>
      <!-- Check if this is really a book section, i.e., it has both a book title. -->
      <xsl:when test="string-length($reference/b:BookTitle) = 0">
        <xsl:call-template name="FormatBookReference">
          <xsl:with-param name="reference" select="$reference" />
        </xsl:call-template>
      </xsl:when>
      <!-- This seems to be a genuine book section, let us format it. -->
      <xsl:otherwise>
        <!-- Format the authors, if not available, fall back to the book authors or editors. -->
        <xsl:variable name="authors">
          <xsl:choose>
            <xsl:when test="string-length($reference/b:Author/b:Author) > 0">
              <xsl:call-template name="FormatAuthors">
                <xsl:with-param name="list" select="$reference/b:Author/b:Author" />
                <xsl:with-param name="maxAuthors" select="$MaxNumberOfAuthors" />
              </xsl:call-template>
            </xsl:when>
            <xsl:when test="string-length($reference/b:Author/b:BookAuthor) > 0">
              <xsl:call-template name="FormatAuthors">
                <xsl:with-param name="list" select="$reference/b:Author/b:BookAuthor" />
                <xsl:with-param name="maxAuthors" select="$MaxNumberOfAuthors" />
              </xsl:call-template>
            </xsl:when>
            <xsl:otherwise>
              <xsl:call-template name="FormatEditors">
                <xsl:with-param name="list" select="$reference/b:Author/b:Editor" />
                <xsl:with-param name="maxEditors" select="$MaxNumberOfEditors" />
              </xsl:call-template>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:variable>

        <!-- Format the title, if not available, fall back to the book title. -->
        <xsl:variable name="title">
          <xsl:if test="string-length($reference/b:Title) > 0">
            <xsl:text>"</xsl:text>
            <xsl:value-of select="$reference/b:Title"/>
            <!-- There will be a book title following this, so no need for checks. -->
            <xsl:text>," in </xsl:text>
          </xsl:if>

          <i>
            <xsl:value-of select="$reference/b:BookTitle"/>
          </i>
        </xsl:variable>

        <!-- Format the editors. -->
        <xsl:variable name="editors">
          <!-- Check if the editors were not used yet. -->
          <xsl:if test="(string-length($reference/b:Author/b:Author) > 0 or string-length($reference/b:Author/b:BookAuthor)) and string-length($reference/b:Author/b:Editor) > 0">
            <xsl:call-template name="FormatEditors">
              <xsl:with-param name="list" select="$reference/b:Author/b:Editor" />
              <xsl:with-param name="maxEditors" select="$MaxNumberOfEditors" />
            </xsl:call-template>
          </xsl:if>
        </xsl:variable>

        <!-- Format the location. -->
        <xsl:variable name="location">
          <xsl:if test="string-length($reference/b:City) >0">
            <xsl:value-of select="$reference/b:City" />
            <xsl:if test="string-length($reference/b:StateProvince) >0 or string-length($reference/b:CountryRegion) > 0">
              <xsl:text>, </xsl:text>
            </xsl:if>
          </xsl:if>

          <xsl:if test="string-length($reference/b:StateProvince) >0">
            <xsl:value-of select="$reference/b:StateProvince" />
            <xsl:if test="string-length($reference/b:CountryRegion) > 0">
              <xsl:text>, </xsl:text>
            </xsl:if>
          </xsl:if>

          <xsl:value-of select="$reference/b:CountryRegion" />
        </xsl:variable>

        <!-- Format the publisher. -->
        <xsl:variable name="publisher">
          <xsl:value-of select="$reference/b:Publisher"/>
        </xsl:variable>

        <!-- Format the date. -->
        <xsl:variable name="date">
          <xsl:call-template name="FormatDate">
            <xsl:with-param name="month" select="$reference/b:Month" />
            <xsl:with-param name="year" select="$reference/b:Year" />
          </xsl:call-template>
        </xsl:variable>

        <!-- Format the volume. -->
        <xsl:variable name="volume">
          <xsl:if test="string-length($reference/b:Volume) > 0">
            <xsl:text>vol. </xsl:text>
            <xsl:value-of select="$reference/b:Volume" />
          </xsl:if>
        </xsl:variable>

        <!-- Format the chapter. -->
        <xsl:variable name="chapter">
          <xsl:if test="string-length($reference/b:ChapterNumber) > 0">
            <xsl:text>ch. </xsl:text>
            <xsl:value-of select="$reference/b:ChapterNumber"/>
          </xsl:if>
        </xsl:variable>

        <!-- Format the pages. -->
        <xsl:variable name="pages">
          <xsl:call-template name="FormatPages">
            <xsl:with-param name="pages" select="$reference/b:Pages" />
          </xsl:call-template>
        </xsl:variable>

        <!-- Display the authors. -->
        <xsl:if test="string-length($authors) > 0">
          <xsl:value-of select="$authors"/>
          <xsl:if test="string-length($title) > 0 or string-length($editors) > 0 or string-length($location) > 0 or string-length($publisher) > 0 or string-length($date) > 0 or string-length($volume) > 0 or string-length($chapter) > 0">
            <xsl:text>, </xsl:text>
          </xsl:if>
        </xsl:if>

        <!-- Display the full title. -->
        <xsl:if test="string-length($title) > 0">
          <xsl:copy-of select="$title"/>
        </xsl:if>

        <!-- Display the editors. -->
        <xsl:choose>
          <xsl:when test="string-length($editors) > 0">
            <xsl:text>, </xsl:text>
            <xsl:value-of select="$editors"/>
            <xsl:if test="string-length($location) > 0 or string-length($publisher) > 0 or string-length($date) > 0 or string-length($volume) > 0 or string-length($chapter) > 0 or string-length($pages) > 0">
              <xsl:text> </xsl:text>
            </xsl:if>
          </xsl:when>
          <xsl:when test="string-length($location) = 0 and string-length($publisher) = 0 and (string-length($date) > 0 or string-length($volume) > 0 or string-length($chapter) > 0 or string-length($pages) > 0)">
            <xsl:text>, </xsl:text>
          </xsl:when>
          <xsl:otherwise>
            <xsl:if test="substring($title, string-length($title) , 1) != '.'">
              <xsl:text>.</xsl:text>
            </xsl:if>
            <xsl:text> </xsl:text>
          </xsl:otherwise>
        </xsl:choose>

        <!-- Display location and publisher. -->
        <xsl:if test="string-length($location) > 0">
          <xsl:value-of select="$location" />
          <xsl:choose>
            <xsl:when test="string-length($publisher) > 0">
              <xsl:text>: </xsl:text>
            </xsl:when>
            <xsl:when test="string-length($date) > 0 or string-length($volume) > 0 or string-length($chapter) > 0 or string-length($pages) > 0">
              <xsl:text>, </xsl:text>
            </xsl:when>
          </xsl:choose>
        </xsl:if>

        <!-- Display the publisher. -->
        <xsl:if test="string-length($publisher) > 0">
          <xsl:value-of select="$publisher"/>
          <xsl:if test="string-length($date) > 0 or string-length($volume) > 0 or string-length($chapter) > 0 or string-length($pages) > 0">
            <xsl:text>, </xsl:text>
          </xsl:if>
        </xsl:if>

        <!-- Display the date. -->
        <xsl:if test="string-length($date) > 0">
          <xsl:value-of select="$date"/>
          <xsl:if test="string-length($volume) > 0 or string-length($chapter) > 0 or string-length($pages) > 0">
            <xsl:text>, </xsl:text>
          </xsl:if>
        </xsl:if>

        <!-- Display the volume. -->
        <xsl:if test="string-length($volume) > 0">
          <xsl:value-of select="$volume"/>
          <xsl:if test="string-length($chapter) > 0 or string-length($pages) > 0">
            <xsl:text>, </xsl:text>
          </xsl:if>
        </xsl:if>

        <!-- Display the chapter number. -->
        <xsl:if test="string-length($chapter) > 0">
          <xsl:value-of select="$chapter"/>
          <xsl:if test="string-length($pages) > 0">
            <xsl:text>, </xsl:text>
          </xsl:if>
        </xsl:if>

        <!-- Display the page numbers. -->
        <xsl:if test="string-length($pages) > 0">
          <xsl:value-of select="$pages"/>
        </xsl:if>

        <!-- Display comments. -->
        <xsl:if test="string-length($reference/b:Comments) > 0">
          <xsl:text>, </xsl:text>
          <xsl:value-of select="$reference/b:Comments"/>
        </xsl:if>

        <!-- Display final '.' -->
        <xsl:if test="string-length($authors) > 0 or string-length($title) > 0 or string-length($editors) > 0 or string-length($location) > 0 or string-length($publisher) > 0 or string-length($date) > 0 or string-length($volume) > 0 or string-length($chapter) > 0">
          <xsl:text>.</xsl:text>
          <xsl:if test="string-length($reference/b:URL) > 0">
            <xsl:text> </xsl:text>
          </xsl:if>
        </xsl:if>

        <!-- Display online availability. -->
        <xsl:if test="string-length($reference/b:URL) > 0">
          <xsl:call-template name="FormatURL">
            <xsl:with-param name="url" select="$reference/b:URL" />
            <xsl:with-param name="displayAsLink" select="$DisplayURLAsLink" />
          </xsl:call-template>
        </xsl:if>

      </xsl:otherwise>
    </xsl:choose>

  </xsl:template>

  <!-- End of templates for BookSection. -->

  <!-- Begin of templates for JournalArticle. -->

  <xsl:template name="SelectImportantFieldsForJournalArticle">
    <b:ImportantFields>
      <b:ImportantField>
        <xsl:text>b:Author/b:Author/b:NameList</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Title</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:JournalName</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Volume</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Issue</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Pages</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Month</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Year</xsl:text>
      </b:ImportantField>
    </b:ImportantFields>
  </xsl:template>

  <xsl:template name="FormatJournalArticleReference">
    <xsl:param name="reference" />

    <!-- Format the authors. -->
    <xsl:variable name="authors">
      <xsl:call-template name="FormatAuthors">
        <xsl:with-param name="list" select="$reference/b:Author/b:Author" />
        <xsl:with-param name="maxAuthors" select="$MaxNumberOfAuthors" />
      </xsl:call-template>
    </xsl:variable>

    <!-- Format the date. -->
    <xsl:variable name="date">
      <xsl:call-template name="FormatDate">
        <xsl:with-param name="month" select="$reference/b:Month" />
        <xsl:with-param name="year" select="$reference/b:Year" />
      </xsl:call-template>
    </xsl:variable>

    <!-- Display the authors. -->
    <xsl:if test="string-length($authors) > 0">
      <xsl:value-of select="$authors"/>
      <xsl:if test="string-length($reference/b:Title) > 0 or string-length($reference/b:JournalName) > 0 or string-length($reference/b:Volume) > 0 or string-length($reference/b:Issue) > 0 or string-length($reference/b:Pages) > 0 or string-length($date) > 0">
        <xsl:text>, </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the article title. -->
    <xsl:if test="string-length($reference/b:Title) > 0">
      <xsl:text>"</xsl:text>
      <xsl:value-of select="$reference/b:Title"/>
      <xsl:if test="string-length($reference/b:JournalName) > 0 or string-length($reference/b:Volume) > 0 or string-length($reference/b:Issue) > 0 or string-length($reference/b:Pages) > 0 or string-length($date) > 0">
        <xsl:text>,</xsl:text>
      </xsl:if>
      <xsl:text>"</xsl:text>
      <xsl:if test="string-length($reference/b:JournalName) > 0 or string-length($reference/b:Volume) > 0 or string-length($reference/b:Issue) > 0 or string-length($reference/b:Pages) > 0 or string-length($date) > 0">
        <xsl:text> </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the journal name. -->
    <xsl:if test="string-length($reference/b:JournalName) > 0">
      <i>
        <xsl:value-of select="$reference/b:JournalName"/>
      </i>
      <xsl:if test="string-length($reference/b:Volume) > 0 or string-length($reference/b:Issue) > 0 or string-length($reference/b:Pages) > 0 or string-length($date) > 0">
        <xsl:text>, </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the volume. -->
    <xsl:if test="string-length($reference/b:Volume) > 0">
      <xsl:text>vol. </xsl:text>
      <xsl:value-of select="$reference/b:Volume"/>
      <xsl:if test="string-length($reference/b:Issue) > 0 or string-length($reference/b:Pages) > 0 or string-length($date) > 0">
        <xsl:text>, </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the issue. -->
    <xsl:if test="string-length($reference/b:Issue) > 0">
      <xsl:text>no. </xsl:text>
      <xsl:value-of select="$reference/b:Issue"/>
      <xsl:if test="string-length($reference/b:Pages) > 0 or string-length($date) > 0">
        <xsl:text>, </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the page numbers. -->
    <xsl:if test="string-length($reference/b:Pages) > 0">
      <xsl:call-template name="FormatPages">
        <xsl:with-param name="pages" select="$reference/b:Pages" />
      </xsl:call-template>
      <xsl:if test="string-length($date) > 0">
        <xsl:text>, </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the date. -->
    <xsl:if test="string-length($date) > 0">
      <xsl:value-of select="$date"/>
    </xsl:if>

    <!-- Display comments. -->
    <xsl:if test="string-length($reference/b:Comments) > 0">
      <xsl:text>, </xsl:text>
      <xsl:value-of select="$reference/b:Comments"/>
    </xsl:if>

    <!-- Display final '.' -->
    <xsl:if test="string-length($authors) > 0 or string-length($reference/b:Title) > 0 or string-length($reference/b:JournalName) > 0 or string-length($reference/b:Volume) > 0 or string-length($reference/b:Issue) > 0 or string-length($reference/b:Pages) > 0 or string-length($date) > 0">
      <xsl:text>.</xsl:text>
      <xsl:if test="string-length($reference/b:URL) > 0">
        <xsl:text> </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display online availability. -->
    <xsl:if test="string-length($reference/b:URL) > 0">
      <xsl:call-template name="FormatURL">
        <xsl:with-param name="url" select="$reference/b:URL" />
        <xsl:with-param name="displayAsLink" select="$DisplayURLAsLink" />
      </xsl:call-template>
    </xsl:if>

  </xsl:template>

  <!-- End of templates for JournalArticle. -->

  <!-- Begin of templates for ArticleInAPeriodical. -->

  <xsl:template name="SelectImportantFieldsForArticleInAPeriodical">
    <b:ImportantFields>
      <b:ImportantField>
        <xsl:text>b:Author/b:Author/b:NameList</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Title</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:PeriodicalTitle</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Volume</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Issue</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Pages</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Month</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Year</xsl:text>
      </b:ImportantField>
    </b:ImportantFields>
  </xsl:template>

  <xsl:template name="FormatArticleInAPeriodicalReference">
    <xsl:param name="reference" />

    <!-- Similar to journal article, but using b:PeriodicalTitle instead of b:JournalName. -->

    <!-- Format the authors. -->
    <xsl:variable name="authors">
      <xsl:call-template name="FormatAuthors">
        <xsl:with-param name="list" select="$reference/b:Author/b:Author" />
        <xsl:with-param name="maxAuthors" select="$MaxNumberOfAuthors" />
      </xsl:call-template>
    </xsl:variable>

    <!-- Format the date. -->
    <xsl:variable name="date">
      <xsl:call-template name="FormatDate">
        <xsl:with-param name="month" select="$reference/b:Month" />
        <xsl:with-param name="year" select="$reference/b:Year" />
      </xsl:call-template>
    </xsl:variable>

    <!-- Display the authors. -->
    <xsl:if test="string-length($authors) > 0">
      <xsl:value-of select="$authors"/>
      <xsl:if test="string-length($reference/b:Title) > 0 or string-length($reference/b:PeriodicalTitle) > 0 or string-length($reference/b:Volume) > 0 or string-length($reference/b:Issue) > 0 or string-length($reference/b:Pages) > 0 or string-length($date) > 0">
        <xsl:text>, </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the article title. -->
    <xsl:if test="string-length($reference/b:Title) > 0">
      <xsl:text>"</xsl:text>
      <xsl:value-of select="$reference/b:Title"/>
      <xsl:if test="string-length($reference/b:PeriodicalTitle) > 0 or string-length($reference/b:Volume) > 0 or string-length($reference/b:Issue) > 0 or string-length($reference/b:Pages) > 0 or string-length($date) > 0">
        <xsl:text>,</xsl:text>
      </xsl:if>
      <xsl:text>"</xsl:text>
      <xsl:if test="string-length($reference/b:PeriodicalTitle) > 0 or string-length($reference/b:Volume) > 0 or string-length($reference/b:Issue) > 0 or string-length($reference/b:Pages) > 0 or string-length($date) > 0">
        <xsl:text> </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the title of the periodical. -->
    <xsl:if test="string-length($reference/b:PeriodicalTitle) > 0">
      <i>
        <xsl:value-of select="$reference/b:PeriodicalTitle"/>
      </i>
      <xsl:if test="string-length($reference/b:Volume) > 0 or string-length($reference/b:Issue) > 0 or string-length($reference/b:Pages) > 0 or string-length($date) > 0">
        <xsl:text>, </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the volume. -->
    <xsl:if test="string-length($reference/b:Volume) > 0">
      <xsl:text>vol. </xsl:text>
      <xsl:value-of select="$reference/b:Volume"/>
      <xsl:if test="string-length($reference/b:Issue) > 0 or string-length($reference/b:Pages) > 0 or string-length($date) > 0">
        <xsl:text>, </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the issue. -->
    <xsl:if test="string-length($reference/b:Issue) > 0">
      <xsl:text>no. </xsl:text>
      <xsl:value-of select="$reference/b:Issue"/>
      <xsl:if test="string-length($reference/b:Pages) > 0 or string-length($date) > 0">
        <xsl:text>, </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the page numbers. -->
    <xsl:if test="string-length($reference/b:Pages) > 0">
      <xsl:call-template name="FormatPages">
        <xsl:with-param name="pages" select="$reference/b:Pages" />
      </xsl:call-template>
      <xsl:if test="string-length($date) > 0">
        <xsl:text>, </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the date. -->
    <xsl:if test="string-length($date) > 0">
      <xsl:value-of select="$date"/>
    </xsl:if>

    <!-- Display comments. -->
    <xsl:if test="string-length($reference/b:Comments) > 0">
      <xsl:text>, </xsl:text>
      <xsl:value-of select="$reference/b:Comments"/>
    </xsl:if>

    <!-- Display final '.' -->
    <xsl:if test="string-length($authors) > 0 or string-length($reference/b:Title) > 0 or string-length($reference/b:PeriodicalTitle) > 0 or string-length($reference/b:Volume) > 0 or string-length($reference/b:Issue) > 0 or string-length($reference/b:Pages) > 0 or string-length($date) > 0">
      <xsl:text>.</xsl:text>
      <xsl:if test="string-length($reference/b:URL) > 0">
        <xsl:text> </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display online availability. -->
    <xsl:if test="string-length($reference/b:URL) > 0">
      <xsl:call-template name="FormatURL">
        <xsl:with-param name="url" select="$reference/b:URL" />
        <xsl:with-param name="displayAsLink" select="$DisplayURLAsLink" />
      </xsl:call-template>
    </xsl:if>

  </xsl:template>

  <!-- End of templates for ArticleInAPeriodical. -->

  <!-- Begin of templates for ConferenceProceedings. -->

  <xsl:template name="SelectImportantFieldsForConferenceProceedings">
    <b:ImportantFields>
      <b:ImportantField>
        <xsl:text>b:Author/b:Author/b:NameList</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Title</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:ConferenceName</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:City</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:CountryRegion</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Month</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Year</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Pages</xsl:text>
      </b:ImportantField>
    </b:ImportantFields>
  </xsl:template>

  <xsl:template name="FormatConferenceProceedingsReference">
    <xsl:param name="reference" />

    <!-- Format authors. -->
    <xsl:variable name="authors">
      <xsl:call-template name="FormatAuthors">
        <xsl:with-param name="list" select="$reference/b:Author/b:Author" />
        <xsl:with-param name="maxAuthors" select="$MaxNumberOfAuthors" />
      </xsl:call-template>
    </xsl:variable>

    <!-- Format title. -->
    <xsl:variable name="title">
      <xsl:value-of select="$reference/b:Title"/>
    </xsl:variable>

    <!-- Format proceedings (the italic markup will be stripped. -->
    <xsl:variable name="proceedings">
      <xsl:text>in </xsl:text>
      <i>
        <xsl:value-of select="$reference/b:ConferenceName"/>
      </i>
    </xsl:variable>

    <!-- Format volume. -->
    <xsl:variable name="volume">
      <xsl:if test="string-length($reference/b:Volume) > 0">
        <xsl:text>vol. </xsl:text>
        <xsl:value-of select="$reference/b:Volume"/>
      </xsl:if>
    </xsl:variable>

    <!-- Format location. -->
    <xsl:variable name="location">
      <xsl:if test="string-length($reference/b:City) >0">
        <xsl:value-of select="$reference/b:City" />
        <xsl:if test="string-length($reference/b:StateProvince) >0 or string-length($reference/b:CountryRegion) > 0">
          <xsl:text>, </xsl:text>
        </xsl:if>
      </xsl:if>

      <xsl:if test="string-length($reference/b:StateProvince) >0">
        <xsl:value-of select="$reference/b:StateProvince" />
        <xsl:if test="string-length($reference/b:CountryRegion) > 0">
          <xsl:text>, </xsl:text>
        </xsl:if>
      </xsl:if>

      <xsl:value-of select="$reference/b:CountryRegion" />
    </xsl:variable>

    <!-- Format date. -->
    <xsl:variable name="date">
      <xsl:call-template name="FormatDate">
        <xsl:with-param name="month" select="$reference/b:Month" />
        <xsl:with-param name="year" select="$reference/b:Year" />
      </xsl:call-template>
    </xsl:variable>

    <!-- Format pages. -->
    <xsl:variable name="pages">
      <xsl:call-template name="FormatPages">
        <xsl:with-param name="pages" select="$reference/b:Pages" />
      </xsl:call-template>
    </xsl:variable>

    <!-- Display authors. -->
    <xsl:if test="string-length($authors) > 0">
      <xsl:value-of select="$authors"/>
      <!-- Display a separator if there still is a next element. -->
      <xsl:if test="string-length($title) > 0 or string-length($proceedings) > 0 or string-length($volume) > 0 or string-length($location) > 0 or string-length($date) > 0 or string-length($pages) > 0">
        <xsl:text>, </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display title. -->
    <xsl:if test="string-length($title) > 0">
      <xsl:text>"</xsl:text>
      <xsl:value-of select="$title"/>
      <!-- Display the first part of the separator if there still is a next element. -->
      <xsl:if test="string-length($proceedings) > 0 or string-length($volume) > 0 or string-length($location) > 0 or string-length($date) > 0 or string-length($pages) > 0">
        <xsl:text>,</xsl:text>
      </xsl:if>
      <xsl:text>"</xsl:text>
      <!-- Display the second part of the separator if there still is a next element. -->
      <xsl:if test="string-length($proceedings) > 0 or string-length($volume) > 0 or string-length($location) > 0 or string-length($date) > 0 or string-length($pages) > 0">
        <xsl:text> </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display proceedings. -->
    <xsl:if test="string-length($proceedings) > 0">
      <xsl:copy-of select="$proceedings"/>
      <!-- Display a separator if there still is a next element. -->
      <xsl:if test="string-length($volume) > 0 or string-length($location) > 0 or string-length($date) > 0 or string-length($pages) > 0">
        <xsl:text>, </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display volumes. -->
    <xsl:if test="string-length($volume) > 0">
      <xsl:value-of select="$volume"/>
      <!-- Display a separator if there still is a next element. -->
      <xsl:if test="string-length($location) > 0 or string-length($date) > 0 or string-length($pages) > 0">
        <xsl:text>, </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display conference location. -->
    <xsl:if test="string-length($location) > 0">
      <xsl:value-of select="$location"/>
      <!-- Display a separator if there still is a next element. -->
      <xsl:if test="string-length($date) > 0 or string-length($pages) > 0">
        <xsl:text>, </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display conference date. -->
    <xsl:if test="string-length($date) > 0">
      <xsl:value-of select="$date"/>
      <!-- Display a separator if there still is a next element. -->
      <xsl:if test="string-length($pages) > 0">
        <xsl:text>, </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the pages. -->
    <xsl:value-of select="$pages"/>

    <!-- Display final '.' -->
    <xsl:if test="string-length($authors) > 0 or string-length($title) > 0 or string-length($proceedings) > 0 or string-length($volume) > 0 or string-length($location) > 0 or string-length($date) > 0 or string-length($pages) > 0">
      <xsl:text>.</xsl:text>
      <xsl:if test="string-length($reference/b:URL) > 0">
        <xsl:text> </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display online availability. -->
    <xsl:if test="string-length($reference/b:URL) > 0">
      <xsl:call-template name="FormatURL">
        <xsl:with-param name="url" select="$reference/b:URL" />
        <xsl:with-param name="displayAsLink" select="$DisplayURLAsLink" />
      </xsl:call-template>
    </xsl:if>

  </xsl:template>

  <!-- End of templates for ConferenceProceedings. -->

  <!-- Begin of templates for Report. -->

  <xsl:template name="SelectImportantFieldsForReport">
    <b:ImportantFields>
      <b:ImportantField>
        <xsl:text>b:Author/b:Author/b:NameList</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Title</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:ThesisType</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Institution</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:City</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:CountryRegion</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:StandardNumber</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Year</xsl:text>
      </b:ImportantField>
    </b:ImportantFields>
  </xsl:template>

  <xsl:template name="FormatReportReference">
    <xsl:param name="reference" />

    <!-- Report type for deciding on formatting. -->
    <!--   Possible values are: 
             - techreport (technical report, default type)
             - masterthesis (master thesis)
             - phdthesis (phd thesis)
             - standard (international standard)
    -->
    <xsl:variable name="type">
      <xsl:choose>
        <!-- If no type is given, a technical report is assumed. -->
        <xsl:when test="string-length($reference/b:ThesisType) = 0">
          <xsl:text>other</xsl:text>
        </xsl:when>
        <xsl:otherwise>

          <!-- Convert the type to lower case and remove all non-letters. -->
          <xsl:variable name="stripped" select="translate(translate($reference/b:ThesisType, $upper, $lower), ' ,?;.:/=+~-_)}{!(#&#160;', '')" />

          <xsl:choose>
            <xsl:when test="contains($stripped, 'rep')">
              <xsl:text>techreport</xsl:text>
            </xsl:when>
            <xsl:when test="contains($stripped, 'phd')">
              <xsl:text>phdthesis</xsl:text>
            </xsl:when>
            <xsl:when test="contains($stripped, 'ms') or contains($stripped, 'master') or (contains($stripped, 'm') and contains($stripped, 'thesis'))">
              <xsl:text>masterthesis</xsl:text>
            </xsl:when>
            <xsl:otherwise>
              <xsl:text>other</xsl:text>
            </xsl:otherwise>
          </xsl:choose>

        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <!-- Formatting of different types of reports based on the retrieved type. -->
    <xsl:choose>
      <!-- Master thesis. -->
      <xsl:when test="$type = 'masterthesis'">
        <!-- Format author info. -->
        <xsl:variable name="authors">
          <xsl:call-template name="FormatAuthors">
            <xsl:with-param name="list" select="$reference/b:Author/b:Author" />
            <xsl:with-param name="maxAuthors" select="$MaxNumberOfAuthors" />
          </xsl:call-template>
        </xsl:variable>

        <!-- Format title. -->
        <xsl:variable name="title">
          <xsl:if test="string-length($reference/b:Title) > 0">
            <xsl:text>"</xsl:text>
            <xsl:value-of select="$reference/b:Title"/>
            <xsl:text>," </xsl:text>
          </xsl:if>
          <xsl:choose>
            <xsl:when test="string-length($reference/b:ThesisType) > 0">
              <xsl:value-of select="$reference/b:ThesisType"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:text>Master's thesis</xsl:text>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:variable>

        <!-- Format the institution and its location. -->
        <xsl:variable name="institution">
          <xsl:if test="string-length($reference/b:Department) > 0">
            <xsl:value-of select="$reference/b:Department"/>
            <xsl:if test="string-length($reference/b:Institution) > 0 or string-length($reference/b:City) > 0 or string-length($reference/b:StateProvince) >0 or string-length($reference/b:CountryRegion) > 0" >
              <xsl:text>, </xsl:text>
            </xsl:if>
          </xsl:if>

          <xsl:if test="string-length($reference/b:Institution) > 0">
            <xsl:value-of select="$reference/b:Institution"/>
            <xsl:if test="string-length($reference/b:City) > 0 or string-length($reference/b:StateProvince) >0 or string-length($reference/b:CountryRegion) > 0" >
              <xsl:text>, </xsl:text>
            </xsl:if>
          </xsl:if>

          <xsl:if test="string-length($reference/b:City) >0">
            <xsl:value-of select="$reference/b:City" />
            <xsl:if test="string-length($reference/b:StateProvince) >0 or string-length($reference/b:CountryRegion) > 0">
              <xsl:text>, </xsl:text>
            </xsl:if>
          </xsl:if>

          <xsl:if test="string-length($reference/b:StateProvince) >0">
            <xsl:value-of select="$reference/b:StateProvince" />
            <xsl:if test="string-length($reference/b:CountryRegion) > 0">
              <xsl:text>, </xsl:text>
            </xsl:if>
          </xsl:if>

          <xsl:value-of select="$reference/b:CountryRegion" />
        </xsl:variable>

        <!-- Format a date (Month Day, Year). -->
        <xsl:variable name="date">
          <xsl:call-template name="FormatDate">
            <xsl:with-param name="day" select="$reference/b:Day" />
            <xsl:with-param name="month" select="$reference/b:Month" />
            <xsl:with-param name="year" select="$reference/b:Year" />
          </xsl:call-template>
        </xsl:variable>

        <!-- Format an URL. -->
        <xsl:variable name="url">
          <xsl:if test="string-length($reference/b:URL) > 0">
            <xsl:call-template name="FormatURL">
              <xsl:with-param name="url" select="$reference/b:URL" />
              <xsl:with-param name="displayAsLink" select="$DisplayURLAsLink" />
            </xsl:call-template>
          </xsl:if>
        </xsl:variable>

        <!-- Display authors. -->
        <xsl:if test="string-length($authors) > 0">
          <xsl:value-of select="$authors"/>
          <xsl:text>, </xsl:text>
        </xsl:if>

        <!-- Display title.-->
        <xsl:value-of select="$title"/>

        <!-- Display institution. -->
        <xsl:if test="string-length($institution) > 0">
          <!-- There was a title! -->
          <xsl:text>, </xsl:text>
          <xsl:value-of select="$institution"/>
        </xsl:if>

        <!-- Display date of thesis. -->
        <xsl:if test="string-length($date) > 0">
          <xsl:text>, </xsl:text>
          <xsl:value-of select="$date"/>
        </xsl:if>

        <!-- Final period (there always was a title, so '.' is necessary. -->
        <xsl:text>.</xsl:text>

        <!-- Display online availability. -->
        <xsl:if test="string-length($url) > 0">
          <xsl:text> </xsl:text>
          <xsl:value-of select="$url" disable-output-escaping="yes" />
        </xsl:if>

      </xsl:when>

      <!-- PhD thesis. -->
      <xsl:when test="$type = 'phdthesis'">
        <!-- Format author info. -->
        <xsl:variable name="authors">
          <xsl:call-template name="FormatAuthors">
            <xsl:with-param name="list" select="$reference/b:Author/b:Author" />
            <xsl:with-param name="maxAuthors" select="$MaxNumberOfAuthors" />
          </xsl:call-template>
        </xsl:variable>

        <!-- Format title. -->
        <xsl:variable name="title">
          <xsl:if test="string-length($reference/b:Title) > 0">
            <xsl:text>"</xsl:text>
            <xsl:value-of select="$reference/b:Title"/>
            <xsl:text>," </xsl:text>
          </xsl:if>
          <xsl:choose>
            <xsl:when test="string-length($reference/b:ThesisType) > 0">
              <xsl:value-of select="$reference/b:ThesisType"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:text>Ph.D. dissertation</xsl:text>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:variable>

        <!-- Format the institution and its location. -->
        <xsl:variable name="institution">
          <xsl:if test="string-length($reference/b:Department) > 0">
            <xsl:value-of select="$reference/b:Department"/>
            <xsl:if test="string-length($reference/b:Institution) > 0 or string-length($reference/b:City) > 0 or string-length($reference/b:StateProvince) >0 or string-length($reference/b:CountryRegion) > 0" >
              <xsl:text>, </xsl:text>
            </xsl:if>
          </xsl:if>

          <xsl:if test="string-length($reference/b:Institution) > 0">
            <xsl:value-of select="$reference/b:Institution"/>
            <xsl:if test="string-length($reference/b:City) > 0 or string-length($reference/b:StateProvince) >0 or string-length($reference/b:CountryRegion) > 0" >
              <xsl:text>, </xsl:text>
            </xsl:if>
          </xsl:if>

          <xsl:if test="string-length($reference/b:City) >0">
            <xsl:value-of select="$reference/b:City" />
            <xsl:if test="string-length($reference/b:StateProvince) >0 or string-length($reference/b:CountryRegion) > 0">
              <xsl:text>, </xsl:text>
            </xsl:if>
          </xsl:if>

          <xsl:if test="string-length($reference/b:StateProvince) >0">
            <xsl:value-of select="$reference/b:StateProvince" />
            <xsl:if test="string-length($reference/b:CountryRegion) > 0">
              <xsl:text>, </xsl:text>
            </xsl:if>
          </xsl:if>

          <xsl:value-of select="$reference/b:CountryRegion" />
        </xsl:variable>

        <!-- Format a date (Month Day, Year). -->
        <xsl:variable name="date">
          <xsl:call-template name="FormatDate">
            <xsl:with-param name="day" select="$reference/b:Day" />
            <xsl:with-param name="month" select="$reference/b:Month" />
            <xsl:with-param name="year" select="$reference/b:Year" />
          </xsl:call-template>
        </xsl:variable>

        <!-- Format an URL. -->
        <xsl:variable name="url">
          <xsl:if test="string-length($reference/b:URL) > 0">
            <xsl:call-template name="FormatURL">
              <xsl:with-param name="url" select="$reference/b:URL" />
              <xsl:with-param name="displayAsLink" select="$DisplayURLAsLink" />
            </xsl:call-template>
          </xsl:if>
        </xsl:variable>

        <!-- Display authors. -->
        <xsl:if test="string-length($authors) > 0">
          <xsl:value-of select="$authors"/>
          <xsl:text>, </xsl:text>
        </xsl:if>

        <!-- Display title.-->
        <xsl:value-of select="$title"/>

        <!-- Display institution. -->
        <xsl:if test="string-length($institution) > 0">
          <!-- There was a title! -->
          <xsl:text>, </xsl:text>
          <xsl:value-of select="$institution"/>
        </xsl:if>

        <!-- Display date of thesis. -->
        <xsl:if test="string-length($date) > 0">
          <xsl:text>, </xsl:text>
          <xsl:value-of select="$date"/>
        </xsl:if>

        <!-- Final period (there always was a title, so '.' is necessary. -->
        <xsl:text>.</xsl:text>

        <!-- Display online availability. -->
        <xsl:if test="string-length($url) > 0">
          <xsl:text> </xsl:text>
          <xsl:value-of select="$url" disable-output-escaping="yes" />
        </xsl:if>

      </xsl:when>

      <!-- Technical report. -->
      <xsl:when test="$type = 'techreport'">
        <!-- Format author info. -->
        <xsl:variable name="authors">
          <xsl:call-template name="FormatAuthors">
            <xsl:with-param name="list" select="$reference/b:Author/b:Author" />
            <xsl:with-param name="maxAuthors" select="$MaxNumberOfAuthors" />
          </xsl:call-template>
        </xsl:variable>

        <!-- Format title. -->
        <xsl:variable name="title">
          <xsl:if test="string-length($reference/b:Title) > 0">
            <xsl:text>"</xsl:text>
            <xsl:value-of select="$reference/b:Title"/>
            <xsl:text>," </xsl:text>
          </xsl:if>
        </xsl:variable>

        <!-- Format the institution and its location. -->
        <xsl:variable name="institution">
          <xsl:if test="string-length($reference/b:Department) > 0">
            <xsl:value-of select="$reference/b:Department"/>
            <xsl:if test="string-length($reference/b:Institution) > 0 or string-length($reference/b:City) > 0 or string-length($reference/b:StateProvince) >0 or string-length($reference/b:CountryRegion) > 0" >
              <xsl:text>, </xsl:text>
            </xsl:if>
          </xsl:if>

          <xsl:if test="string-length($reference/b:Institution) > 0">
            <xsl:value-of select="$reference/b:Institution"/>
            <xsl:if test="string-length($reference/b:City) > 0 or string-length($reference/b:StateProvince) >0 or string-length($reference/b:CountryRegion) > 0" >
              <xsl:text>, </xsl:text>
            </xsl:if>
          </xsl:if>

          <xsl:if test="string-length($reference/b:City) >0">
            <xsl:value-of select="$reference/b:City" />
            <xsl:if test="string-length($reference/b:StateProvince) >0 or string-length($reference/b:CountryRegion) > 0">
              <xsl:text>, </xsl:text>
            </xsl:if>
          </xsl:if>

          <xsl:if test="string-length($reference/b:StateProvince) >0">
            <xsl:value-of select="$reference/b:StateProvince" />
            <xsl:if test="string-length($reference/b:CountryRegion) > 0">
              <xsl:text>, </xsl:text>
            </xsl:if>
          </xsl:if>

          <xsl:value-of select="$reference/b:CountryRegion" />
        </xsl:variable>

        <!-- Format the report type and number. -->
        <xsl:variable name="report">
          <xsl:choose>
            <xsl:when test="string-length($reference/b:ThesisType) > 0">
              <xsl:value-of select="$reference/b:ThesisType"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:text>Tech. Rep.</xsl:text>
            </xsl:otherwise>
          </xsl:choose>
          <xsl:if test="string-length($reference/b:StandardNumber) > 0">
            <xsl:text> </xsl:text>
            <xsl:value-of select="$reference/b:StandardNumber"/>
          </xsl:if>
        </xsl:variable>

        <!-- Format a date (Month Day, Year). -->
        <xsl:variable name="date">
          <xsl:call-template name="FormatDate">
            <xsl:with-param name="day" select="$reference/b:Day" />
            <xsl:with-param name="month" select="$reference/b:Month" />
            <xsl:with-param name="year" select="$reference/b:Year" />
          </xsl:call-template>
        </xsl:variable>

        <!-- Format an URL. -->
        <xsl:variable name="url">
          <xsl:if test="string-length($reference/b:URL) > 0">
            <xsl:call-template name="FormatURL">
              <xsl:with-param name="url" select="$reference/b:URL" />
              <xsl:with-param name="displayAsLink" select="$DisplayURLAsLink" />
            </xsl:call-template>
          </xsl:if>
        </xsl:variable>

        <!-- Display authors. -->
        <xsl:if test="string-length($authors) > 0">
          <xsl:value-of select="$authors"/>
        </xsl:if>

        <!-- Display title.-->
        <xsl:if test="string-length($title) > 0">
          <xsl:if test="string-length($authors) > 0">
            <xsl:text>, </xsl:text>
          </xsl:if>
          <xsl:value-of select="$title"/>
        </xsl:if>

        <!-- Display institution. -->
        <xsl:if test="string-length($institution) > 0">
          <xsl:value-of select="$institution"/>
        </xsl:if>

        <!-- Display report information. -->
        <xsl:if test="string-length($report) > 0">
          <xsl:if test="string-length($institution) > 0">
            <xsl:text>, </xsl:text>
          </xsl:if>
          <xsl:value-of select="$report"/>
        </xsl:if>

        <!-- Display date of thesis. -->
        <xsl:if test="string-length($date) > 0">
          <xsl:if test="string-length($institution) > 0 or string-length($report) > 0">
            <xsl:text>, </xsl:text>
          </xsl:if>
          <xsl:value-of select="$date"/>
        </xsl:if>

        <!-- Final period (there always was a report type, so '.' is necessary. -->
        <xsl:text>.</xsl:text>

        <!-- Display online availability. -->
        <xsl:if test="string-length($url) > 0">
          <xsl:text> </xsl:text>
          <xsl:value-of select="$url" disable-output-escaping="yes" />
        </xsl:if>

      </xsl:when>

      <!-- Other (Standard, ...). -->
      <xsl:when test="$type = 'other'">
        <!-- Format author info. -->
        <xsl:variable name="authors">
          <xsl:call-template name="FormatAuthors">
            <xsl:with-param name="list" select="$reference/b:Author/b:Author" />
            <xsl:with-param name="maxAuthors" select="$MaxNumberOfAuthors" />
          </xsl:call-template>
        </xsl:variable>
        
        <!-- Format title. -->
        <xsl:variable name="title">          
          <xsl:value-of select="$reference/b:Title"/>         
        </xsl:variable>

        <!-- Format report info. -->
        <xsl:variable name="report">
          <xsl:if test="string-length($reference/b:Institution) > 0">
            <xsl:value-of select="$reference/b:Institution"/>
            <xsl:if test="string-length($reference/b:ThesisType) > 0 or string-length($reference/b:StandardNumber) > 0">
              <xsl:text> </xsl:text>
            </xsl:if>
          </xsl:if>
          
          <xsl:if test="string-length($reference/b:ThesisType) > 0">
            <xsl:value-of select="$reference/b:ThesisType"/>
            <xsl:if test="string-length($reference/b:StandardNumber) > 0">
              <xsl:text> </xsl:text>
            </xsl:if>
          </xsl:if>

          <xsl:if test="string-length($reference/b:StandardNumber) > 0">
            <xsl:value-of select="$reference/b:StandardNumber"/>
          </xsl:if>
          
        </xsl:variable>
        
        <!-- Format date. -->
        <xsl:variable name="date">
          <xsl:call-template name="FormatDate">
            <xsl:with-param name="month" select="$reference/b:Month" />
            <xsl:with-param name="year" select="$reference/b:Year" />
          </xsl:call-template>
        </xsl:variable>

        <!-- Format an URL. -->
        <xsl:variable name="url">
          <xsl:if test="string-length($reference/b:URL) > 0">
            <xsl:call-template name="FormatURL">
              <xsl:with-param name="url" select="$reference/b:URL" />
              <xsl:with-param name="displayAsLink" select="$DisplayURLAsLink" />
            </xsl:call-template>
          </xsl:if>
        </xsl:variable>
        
        <!-- Display authors. -->
        <xsl:if test="string-length($authors) > 0">
          <xsl:value-of select="$authors"/>
          <xsl:if test="string-length($title) > 0 or string-length($report) > 0 or string-length($date) > 0">
            <xsl:text>, </xsl:text>
          </xsl:if>
        </xsl:if>
        
        <!-- Display title. -->
        <xsl:if test="string-length($title) > 0">
          <xsl:text>"</xsl:text>
          <xsl:value-of select="$title"/>
          <xsl:if test="string-length($report) > 0 or string-length($date) > 0">
            <xsl:text>,</xsl:text>
          </xsl:if>
          <xsl:text>"</xsl:text>
          <xsl:if test="string-length($report) > 0 or string-length($date) > 0">
            <xsl:text> </xsl:text>
          </xsl:if>
        </xsl:if>
        
        <!-- Display report. -->
        <xsl:if test="string-length($report) > 0">
          <xsl:value-of select="$report"/>
          <xsl:if test="string-length($date) > 0">
            <xsl:text>, </xsl:text>
          </xsl:if>
        </xsl:if>

        <!-- Display date. -->
        <xsl:if test="string-length($date) > 0">
          <xsl:value-of select="$date"/>
        </xsl:if>

        <!-- Final period (there always was a report type, so '.' is necessary. -->
        <xsl:text>.</xsl:text>

        <!-- Display online availability. -->
        <xsl:if test="string-length($url) > 0">
          <xsl:text> </xsl:text>
          <xsl:value-of select="$url" disable-output-escaping="yes" />
        </xsl:if>

      </xsl:when>
    </xsl:choose>
  </xsl:template>

  <!-- End of templates for Report. -->

  <!-- Begin of templates for InternetSite. -->

  <xsl:template name="SelectImportantFieldsForInternetSite">
    <b:ImportantFields>
      <b:ImportantField>
        <xsl:text>b:Author/b:Author/b:NameList</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:InternetSiteTitle</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Year</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Month</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:URL</xsl:text>
      </b:ImportantField>
    </b:ImportantFields>
  </xsl:template>

  <xsl:template name="FormatInternetSiteReference">
    <xsl:param name="reference" />

    <!-- Format the authors. -->
    <xsl:variable name="authors">
      <xsl:if test="string-length($reference/b:Author/b:Author) > 0">
        <xsl:variable name="temp">
          <xsl:call-template name="FormatAuthors">
            <xsl:with-param name="list" select="$reference/b:Author/b:Author" />
            <xsl:with-param name="maxAuthors" select="$MaxNumberOfAuthors" />
          </xsl:call-template>
        </xsl:variable>

        <xsl:value-of select="$temp"/>
        <xsl:if test="not(substring($temp, string-length($temp), 1) = '.')">
          <xsl:text>.</xsl:text>
        </xsl:if>
      </xsl:if>
    </xsl:variable>

    <!-- Format the date. -->
    <xsl:variable name="date">
      <xsl:if test="string-length($reference/b:Year) > 0">
        <xsl:text>(</xsl:text>
        <xsl:value-of select="$reference/b:Year" />
        <xsl:if test="string-length($reference/b:Month) > 0">
          <xsl:text>, </xsl:text>
          <xsl:call-template name="FormatMonth">
            <xsl:with-param name="month" select="$reference/b:Month" />
          </xsl:call-template>
        </xsl:if>
        <xsl:text>)</xsl:text>
      </xsl:if>
    </xsl:variable>

    <!-- Format the title. -->
    <xsl:variable name="title">
      <xsl:choose>
        <xsl:when test="string-length($reference/b:InternetSiteTitle) > 0">
          <xsl:value-of select="$reference/b:InternetSiteTitle" />
          <xsl:text>.</xsl:text>
        </xsl:when>
        <xsl:when test="string-length($reference/b:Title) > 0">
          <xsl:value-of select="$reference/b:Title" />
          <xsl:text>.</xsl:text>
        </xsl:when>
      </xsl:choose>
    </xsl:variable>

    <!-- Format an URL. -->
    <xsl:variable name="url">
      <xsl:if test="string-length($reference/b:URL) > 0">
        <xsl:call-template name="FormatURL">
          <xsl:with-param name="url" select="$reference/b:URL" />
          <xsl:with-param name="displayAsLink" select="$DisplayURLAsLink" />
        </xsl:call-template>
      </xsl:if>
    </xsl:variable>

    <!-- Display the authors. -->
    <xsl:if test="string-length($authors) > 0">
      <xsl:value-of select="$authors"/>
      <xsl:if test="string-length($date) > 0 or string-length($title) > 0 or string-length($url) > 0">
        <xsl:text> </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the date. -->
    <xsl:if test="string-length($date) > 0">
      <xsl:value-of select="$date"/>
      <xsl:if test="string-length($title) > 0 or string-length($url) > 0">
        <xsl:text> </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the title. -->
    <xsl:if test="string-length($title) > 0">
      <xsl:value-of select="$title"/>
      <xsl:if test="string-length($url) > 0">
        <xsl:text> </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the URL. -->
    <xsl:if test="string-length($url) > 0">
      <xsl:value-of select="$url" disable-output-escaping="yes" />
    </xsl:if>

  </xsl:template>

  <!-- End of templates for InternetSite. -->

  <!-- Begin of templates for DocumentFromInternetSite. -->

  <xsl:template name="SelectImportantFieldsForDocumentFromInternetSite">
    <b:ImportantFields>
      <b:ImportantField>
        <xsl:text>b:Author/b:Author/b:NameList</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:InternetSiteTitle</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Year</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Month</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:URL</xsl:text>
      </b:ImportantField>
    </b:ImportantFields>
  </xsl:template>

  <xsl:template name="FormatDocumentFromInternetSiteReference">
    <xsl:param name="reference" />

    <!-- Format the authors. -->
    <xsl:variable name="authors">
      <xsl:if test="string-length($reference/b:Author/b:Author) > 0">
        <xsl:variable name="temp">
          <xsl:call-template name="FormatAuthors">
            <xsl:with-param name="list" select="$reference/b:Author/b:Author" />
            <xsl:with-param name="maxAuthors" select="$MaxNumberOfAuthors" />
          </xsl:call-template>
        </xsl:variable>

        <xsl:value-of select="$temp"/>
        <xsl:if test="not(substring($temp, string-length($temp), 1) = '.')">
          <xsl:text>.</xsl:text>
        </xsl:if>
      </xsl:if>
    </xsl:variable>

    <!-- Format the date. -->
    <xsl:variable name="date">
      <xsl:if test="string-length($reference/b:Year) > 0">
        <xsl:text>(</xsl:text>
        <xsl:value-of select="$reference/b:Year" />
        <xsl:if test="string-length($reference/b:Month) > 0">
          <xsl:text>, </xsl:text>
          <xsl:call-template name="FormatMonth">
            <xsl:with-param name="month" select="$reference/b:Month" />
          </xsl:call-template>
        </xsl:if>
        <xsl:text>)</xsl:text>
      </xsl:if>
    </xsl:variable>

    <!-- Format the title. -->
    <xsl:variable name="title">
      <xsl:choose>
        <xsl:when test="string-length($reference/b:InternetSiteTitle) > 0">
          <xsl:value-of select="$reference/b:InternetSiteTitle" />
          <xsl:text>.</xsl:text>
        </xsl:when>
        <xsl:when test="string-length($reference/b:Title) > 0">
          <xsl:value-of select="$reference/b:Title" />
          <xsl:text>.</xsl:text>
        </xsl:when>
      </xsl:choose>
    </xsl:variable>

    <!-- Format an URL. -->
    <xsl:variable name="url">
      <xsl:if test="string-length($reference/b:URL) > 0">
        <xsl:call-template name="FormatURL">
          <xsl:with-param name="url" select="$reference/b:URL" />
          <xsl:with-param name="displayAsLink" select="$DisplayURLAsLink" />
        </xsl:call-template>
      </xsl:if>
    </xsl:variable>

    <!-- Display the authors. -->
    <xsl:if test="string-length($authors) > 0">
      <xsl:value-of select="$authors"/>
      <xsl:if test="string-length($date) > 0 or string-length($title) > 0 or string-length($url) > 0">
        <xsl:text> </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the date. -->
    <xsl:if test="string-length($date) > 0">
      <xsl:value-of select="$date"/>
      <xsl:if test="string-length($title) > 0 or string-length($url) > 0">
        <xsl:text> </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the title. -->
    <xsl:if test="string-length($title) > 0">
      <xsl:value-of select="$title"/>
      <xsl:if test="string-length($url) > 0">
        <xsl:text> </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the URL. -->
    <xsl:if test="string-length($url) > 0">
      <xsl:value-of select="$url" disable-output-escaping="yes" />
    </xsl:if>

  </xsl:template>

  <!-- End of templates for DocumentFromInternetSite. -->

  <!-- Begin of templates for ElectronicSource. -->

  <xsl:template name="SelectImportantFieldsForElectronicSource">
    <b:ImportantFields>
      <b:ImportantField>
        <xsl:text>b:Author/b:Author/b:NameList</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Title</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Year</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Month</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Medium</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:URL</xsl:text>
      </b:ImportantField>
    </b:ImportantFields>
  </xsl:template>

  <xsl:template name="FormatElectronicSourceReference">
    <xsl:param name="reference" />

    <!-- Format the authors. -->
    <xsl:variable name="authors">
      <xsl:if test="string-length($reference/b:Author/b:Author) > 0">
        <xsl:variable name="temp">
          <xsl:call-template name="FormatAuthors">
            <xsl:with-param name="list" select="$reference/b:Author/b:Author" />
            <xsl:with-param name="maxAuthors" select="$MaxNumberOfAuthors" />
          </xsl:call-template>
        </xsl:variable>

        <xsl:value-of select="$temp"/>
        <xsl:if test="not(substring($temp, string-length($temp), 1) = '.')">
          <xsl:text>.</xsl:text>
        </xsl:if>
      </xsl:if>
    </xsl:variable>

    <!-- Format the date. -->
    <xsl:variable name="date">
      <xsl:if test="string-length($reference/b:Year) > 0">
        <xsl:text>(</xsl:text>
        <xsl:value-of select="$reference/b:Year" />
        <xsl:if test="string-length($reference/b:Month) > 0">
          <xsl:text>, </xsl:text>
          <xsl:call-template name="FormatMonth">
            <xsl:with-param name="month" select="$reference/b:Month" />
          </xsl:call-template>
        </xsl:if>
        <xsl:text>)</xsl:text>
      </xsl:if>
    </xsl:variable>

    <!-- Format the title. -->
    <xsl:variable name="title">
      <xsl:choose>
        <xsl:when test="string-length($reference/b:Title) > 0">
          <xsl:value-of select="$reference/b:Title" />
          <xsl:text>.</xsl:text>
        </xsl:when>
        <xsl:when test="string-length($reference/b:PublicationTitle) > 0">
          <xsl:value-of select="$reference/b:PublicationTitle" />
          <xsl:text>.</xsl:text>
        </xsl:when>
      </xsl:choose>
    </xsl:variable>

    <!-- Format the type of source. -->
    <xsl:variable name="type">
      <xsl:if test="string-length($reference/b:Medium) > 0">
        <xsl:value-of select="$reference/b:Medium" />
        <xsl:text>.</xsl:text>
      </xsl:if>
    </xsl:variable>

    <!-- Format an URL. -->
    <xsl:variable name="url">
      <xsl:if test="string-length($reference/b:URL) > 0">
        <xsl:call-template name="FormatURL">
          <xsl:with-param name="url" select="$reference/b:URL" />
          <xsl:with-param name="displayAsLink" select="$DisplayURLAsLink" />
        </xsl:call-template>
      </xsl:if>
    </xsl:variable>

    <!-- Display the authors. -->
    <xsl:if test="string-length($authors) > 0">
      <xsl:value-of select="$authors"/>
      <xsl:if test="string-length($date) > 0 or string-length($title) > 0 or string-length($type) > 0 or string-length($url) > 0">
        <xsl:text> </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the date. -->
    <xsl:if test="string-length($date) > 0">
      <xsl:value-of select="$date"/>
      <xsl:if test="string-length($title) > 0 or string-length($type) > 0 or string-length($url) > 0">
        <xsl:text> </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the title. -->
    <xsl:if test="string-length($title) > 0">
      <xsl:value-of select="$title"/>
      <xsl:if test="string-length($type) > 0 or string-length($url) > 0">
        <xsl:text> </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the type. -->
    <xsl:if test="string-length($type) > 0">
      <xsl:value-of select="$type"/>
      <xsl:if test="string-length($url) > 0">
        <xsl:text> </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display the URL. -->
    <xsl:if test="string-length($url) > 0">
      <xsl:value-of select="$url" disable-output-escaping="yes"/>
    </xsl:if>

  </xsl:template>

  <!-- End of templates for ElectronicSource. -->

  <!-- Begin of templates for Art. -->

  <xsl:template name="SelectImportantFieldsForArt">
    <b:ImportantFields>

    </b:ImportantFields>
  </xsl:template>

  <xsl:template name="FormatArtReference">
    <xsl:param name="reference" />

    <xsl:text>Art: TO DO.</xsl:text>
  </xsl:template>

  <!-- End of templates for Art. -->

  <!-- Begin of templates for SoundRecording. -->

  <xsl:template name="SelectImportantFieldsForSoundRecording">
    <b:ImportantFields>

    </b:ImportantFields>
  </xsl:template>

  <xsl:template name="FormatSoundRecordingReference">
    <xsl:param name="reference" />

    <xsl:text>SoundRecording: TO DO.</xsl:text>
  </xsl:template>

  <!-- End of templates for SoundRecording. -->

  <!-- Begin of templates for Performance. -->

  <xsl:template name="SelectImportantFieldsForPerformance">
    <b:ImportantFields>

    </b:ImportantFields>
  </xsl:template>

  <xsl:template name="FormatPerformanceReference">
    <xsl:param name="reference" />

    <xsl:text>Performance: TO DO.</xsl:text>
  </xsl:template>

  <!-- End of templates for Performance. -->

  <!-- Begin of templates for Film. -->

  <xsl:template name="SelectImportantFieldsForFilm">
    <b:ImportantFields>

    </b:ImportantFields>
  </xsl:template>

  <xsl:template name="FormatFilmReference">
    <xsl:param name="reference" />

    <xsl:text>Film: TO DO.</xsl:text>
  </xsl:template>

  <!-- End of templates for Film. -->

  <!-- Begin of templates for Interview. -->

  <xsl:template name="SelectImportantFieldsForInterview">
    <b:ImportantFields>

    </b:ImportantFields>
  </xsl:template>

  <xsl:template name="FormatInterviewReference">
    <xsl:param name="reference" />

    <xsl:text>Interview: TO DO.</xsl:text>
  </xsl:template>

  <!-- End of templates for Interview. -->

  <!-- Begin of templates for Patent. -->

  <xsl:template name="SelectImportantFieldsForPatent">
    <b:ImportantFields>
      <b:ImportantField>
        <xsl:text>b:Author/b:Inventor/b:NameList</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Title</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:CountryRegion</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Type</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:PatentNumber</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Month</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Day</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Year</xsl:text>
      </b:ImportantField>
    </b:ImportantFields>
  </xsl:template>

  <xsl:template name="FormatPatentReference">
    <xsl:param name="reference" />

    <!-- Format the names of the authors of the patent. -->
    <xsl:variable name="authors">
      <xsl:choose>
        <!-- The inventors should be the authors. -->
        <xsl:when test="string-length($reference/b:Author/b:Inventor) > 0">
          <xsl:call-template name="FormatAuthors">
            <xsl:with-param name="list" select="$reference/b:Author/b:Inventor" />
            <xsl:with-param name="maxAuthors" select="$MaxNumberOfAuthors" />
          </xsl:call-template>
        </xsl:when>
        <!-- If the inventors are not the authors, fall back to the old fashioned authors. -->
        <xsl:otherwise>
          <xsl:call-template name="FormatAuthors">
            <xsl:with-param name="list" select="$reference/b:Author/b:Author" />
            <xsl:with-param name="maxAuthors" select="$MaxNumberOfAuthors" />
          </xsl:call-template>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <!-- Format the title of the patent (without enclosing quotes). -->
    <xsl:variable name="title">
      <xsl:value-of select="$reference/b:Title" />
    </xsl:variable>

    <!-- Format the patent information (Country Type Number). -->
    <xsl:variable name="patentinfo">
      <xsl:variable name="temp">
        <xsl:if test="string-length($reference/b:CountryRegion) > 0">
          <xsl:choose>
            <xsl:when test="contains(translate($reference/b:CountryRegion, $upper, $lower),'united states')">
              <xsl:text>U.S. </xsl:text>
            </xsl:when>
            <xsl:otherwise>
              <xsl:call-template name="TranslateCountryToNationality">
                <xsl:with-param name="country" select="$reference/b:CountryRegion" />
              </xsl:call-template>
              <xsl:text> </xsl:text>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:if>

        <xsl:choose>
          <xsl:when test="string-length($reference/b:Type) > 0">
            <xsl:value-of select="$reference/b:Type" />
            <xsl:text> </xsl:text>
          </xsl:when>
          <xsl:otherwise>
            <xsl:text>Patent </xsl:text>
          </xsl:otherwise>
        </xsl:choose>

        <xsl:choose>
          <xsl:when test="string-length($reference/b:PatentNumber) > 0">
            <xsl:value-of select="$reference/b:PatentNumber" />
          </xsl:when>
          <xsl:otherwise>
            <xsl:text>UNKNOWN</xsl:text>
          </xsl:otherwise>
        </xsl:choose>

      </xsl:variable>

      <!-- Remove leading and trailing white space. -->
      <xsl:value-of select="normalize-space($temp)"/>

    </xsl:variable>

    <!-- Format a date (Month Day, Year). -->
    <xsl:variable name="date">
      <xsl:call-template name="FormatDate">
        <xsl:with-param name="day" select="$reference/b:Day" />
        <xsl:with-param name="month" select="$reference/b:Month" />
        <xsl:with-param name="year" select="$reference/b:Year" />
      </xsl:call-template>
    </xsl:variable>

    <!-- Display authors. -->
    <xsl:value-of select="$authors" />

    <!-- Display separator between authors and anything else. -->
    <xsl:if test="string-length($authors) > 0 and (string-length($title) > 0 or string-length($patentinfo) > 0 or string-length($date) > 0)">
      <xsl:text>, </xsl:text>
    </xsl:if>

    <!-- Display title. -->
    <xsl:if test="string-length($title) > 0">
      <xsl:text>"</xsl:text>
      <xsl:value-of select="$title"/>
      <xsl:if test="string-length($patentinfo) > 0 or string-length($date) > 0">
        <xsl:text>,</xsl:text>
      </xsl:if>
      <xsl:text>"</xsl:text>
      <xsl:if test="string-length($patentinfo) > 0 or string-length($date) > 0">
        <xsl:text> </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display patent info. -->
    <xsl:if test="string-length($patentinfo) > 0">
      <xsl:value-of select="$patentinfo"/>
      <xsl:if test="string-length($date) > 0">
        <xsl:text>, </xsl:text>
      </xsl:if>
    </xsl:if>

    <!-- Display date info. -->
    <xsl:value-of select="$date"/>

    <!-- Finish patent. -->
    <xsl:text>.</xsl:text>

  </xsl:template>

  <!-- End of templates for Patent. -->

  <!-- Begin of templates for Case. -->

  <xsl:template name="SelectImportantFieldsForCase">
    <b:ImportantFields>

    </b:ImportantFields>
  </xsl:template>

  <xsl:template name="FormatCaseReference">
    <xsl:param name="reference" />

    <xsl:text>Case: TO DO.</xsl:text>
  </xsl:template>

  <!-- End of templates for Case. -->

  <!-- Begin of templates for Misc. -->

  <xsl:template name="SelectImportantFieldsForMisc">
    <b:ImportantFields>
      <b:ImportantField>
        <xsl:text>b:Author/b:Author/b:NameList</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Title</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Year</xsl:text>
      </b:ImportantField>
      <b:ImportantField>
        <xsl:text>b:Comments</xsl:text>
      </b:ImportantField>
    </b:ImportantFields>
  </xsl:template>

  <xsl:template name="FormatMiscReference">
    <xsl:param name="reference" />

    <xsl:text>Miscellaneous: TO DO.</xsl:text>

  </xsl:template>

  <!-- End of templates for Misc. -->

  <!-- Begin of templates with shared functionality between the different types of references. -->

  <!-- Format authors. -->
  <!--   The parameter 'list' should be either a b:NameList element or a b:Corporate element. -->
  <!--   The parameter 'maxAuthors' should be a number. -->
  <xsl:template name="FormatAuthors">
    <xsl:param name="list" />
    <xsl:param name="maxAuthors" />

    <xsl:choose>
      <!-- In case there are no authors, display whatever there is (most likely a corporate author). -->
      <xsl:when test="count($list/b:NameList) = 0">
        <xsl:value-of select="$list"/>
      </xsl:when>
      <!-- There are authors, format their names correctly. -->
      <xsl:otherwise>
        <xsl:call-template name="FormatNameList">
          <xsl:with-param name="list" select="$list/b:NameList" />
          <xsl:with-param name="maxAuthors" select="$maxAuthors" />
        </xsl:call-template>
      </xsl:otherwise>
    </xsl:choose>

  </xsl:template>

  <!-- Format a list of people. -->
  <!--   The parameter 'list' should be the content of a b:NameList element, i.e. one or more b:Person elements. -->
  <!--   The parameter 'maxAuthors' should be a number. -->
  <xsl:template name="FormatNameList">
    <xsl:param name="list" />
    <xsl:param name="maxAuthors" />

    <xsl:choose>
      <!-- If there are more than the maximum allowed number of authors, only the first author is displayed followed by 'et al.'. -->
      <xsl:when test="count($list/b:Person) > number($maxAuthors)">
        <xsl:call-template name="FormatName">
          <xsl:with-param name="person" select="$list/b:Person[1]"/>
        </xsl:call-template>
        <xsl:text>, et al.</xsl:text>
      </xsl:when>
      <!-- All other cases. -->
      <xsl:otherwise>
        <xsl:for-each select="$list/b:Person">
          <!-- Three locations for a person in the list: first, last, somewhere in the middle. -->
          <!-- If the current person is not the first author and there are more than two authors, add a ', '. -->
          <xsl:if test="position()!=1 and count($list/b:Person) > 2">
            <xsl:text>, </xsl:text>
          </xsl:if>
          <!-- If there are only two persons and this is the last one (space before 'and '). -->
          <xsl:if test="position()= 2 and count($list/b:Person) = 2">
            <xsl:text> </xsl:text>
          </xsl:if>
          <!-- If the current person is the last one but not the first one, add a 'and '. -->
          <xsl:if test="position()!=1 and position() = last()">
            <xsl:text>and </xsl:text>
          </xsl:if>
          <!-- Format the author. -->
          <xsl:call-template name="FormatName">
            <xsl:with-param name="person" select="." />
          </xsl:call-template>
        </xsl:for-each>
      </xsl:otherwise>
    </xsl:choose>

  </xsl:template>

  <!-- Format a name. -->
  <!--   The parameter 'person' should be the content of a b:Person element. -->
  <xsl:template name="FormatName">
    <xsl:param name="person" />

    <!-- Provide an abbreviated version of the first name of a person. -->
    <xsl:variable name="first">
      <xsl:if test="string-length($person/b:First) > 0">
        <xsl:call-template name="AbbreviateName">
          <xsl:with-param name="name" select="$person/b:First"/>
        </xsl:call-template>
      </xsl:if>
    </xsl:variable>

    <!-- Provide an abbreviated version of the middle name of a person. -->
    <xsl:variable name="middle">
      <xsl:if test="string-length($person/b:Middle) > 0">
        <xsl:call-template name="AbbreviateName">
          <xsl:with-param name="name" select="$person/b:Middle"/>
        </xsl:call-template>
      </xsl:if>
    </xsl:variable>

    <!-- The last name of a person, if not give, return 'Unknown' -->
    <xsl:variable name="last">
      <xsl:choose>
        <xsl:when test="string-length($person/b:Last) > 0">
          <xsl:value-of select="$person/b:Last"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:text>Unknown</xsl:text>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <!-- Concatenate the different name parts. -->
    <xsl:choose>
      <xsl:when test="string-length($first) > 0 and string-length($middle) > 0">
        <xsl:value-of select="concat($first,' ',$middle,' ', $last)"/>
      </xsl:when>
      <xsl:when test="string-length($first) > 0 and string-length($middle) = 0">
        <xsl:value-of select="concat($first,' ', $last)"/>
      </xsl:when>
      <xsl:when test="string-length($first) = 0 and string-length($middle) > 0">
        <xsl:value-of select="concat($middle,' ', $last)"/>
      </xsl:when>
      <xsl:when test="string-length($first) = 0 and string-length($middle) = 0">
        <xsl:value-of select="$last"/>
      </xsl:when>
    </xsl:choose>

  </xsl:template>

  <!-- Abbreviate a name. -->
  <xsl:template name="AbbreviateName">
    <xsl:param name="name" />

    <!-- Sanatize name. -->
    <xsl:variable name="sanName">
      <!-- Normalize white space. -->
      <xsl:variable name="t0" select="normalize-space($name)" />
      
      <!-- Handle already abbreviated parts with hypens. -->
      <xsl:variable name="t1">
        <xsl:call-template name="ReplaceText">
          <xsl:with-param name="text" select="$t0" />
          <xsl:with-param name="from" select="'.-'" />
          <xsl:with-param name="to" select="'-'" />
        </xsl:call-template>
      </xsl:variable>

      <!-- Remove with space before hyphens. -->
      <xsl:variable name="t2">
        <xsl:call-template name="ReplaceText">
          <xsl:with-param name="text" select="$name" />
          <xsl:with-param name="from" select="' -'" />
          <xsl:with-param name="to" select="'-'" />
        </xsl:call-template>
      </xsl:variable>

      <!-- Remove with space after hyphens. -->
      <xsl:variable name="t3">
        <xsl:call-template name="ReplaceText">
          <xsl:with-param name="text" select="$name" />
          <xsl:with-param name="from" select="'- '" />
          <xsl:with-param name="to" select="'-'" />
        </xsl:call-template>
      </xsl:variable>
      
      <!-- Replace all remaining dots with a space. -->
      <xsl:variable name="t4" select="translate($t3, '.', ' ')" />

      <!-- Normalize spaces. -->
      <xsl:value-of select="normalize-space($t4)"/>
    </xsl:variable>

    <!-- Abbreviate the sanatized name. -->
    <xsl:call-template name="AbbreviateNamePart">
      <xsl:with-param name="part" select="$sanName" />
      <xsl:with-param name="previousWasChar" select="'false'" />
    </xsl:call-template>
    
  </xsl:template>
  
  <!-- Abbreviate part of a name (recursively called). -->
  <xsl:template name="AbbreviateNamePart">
    <xsl:param name="part" />
    <xsl:param name="previousWasChar" />
    
    <xsl:choose>
      <xsl:when test="string-length($part) > 0">
        <!-- Grab the first character. -->
        <xsl:variable name="char" select="substring($part,1,1)" />

        <xsl:choose>
          <xsl:when test="$char = '-' or $char = ' '">
            <!-- Dot abbreviating this part. -->
            <xsl:text>.</xsl:text>
            <!-- Separator with next part. -->
            <xsl:value-of select="$char"/>
            
            <!-- Abbreviate remainder. -->            
            <xsl:call-template name="AbbreviateNamePart">
              <xsl:with-param name="part" select="substring($part, 2)"/>
              <xsl:with-param name="previousWasChar" select="'false'" />
            </xsl:call-template>
          </xsl:when>
          <xsl:otherwise>
            <xsl:if test="$previousWasChar = 'false'">
              <xsl:value-of select="$char"/>
            </xsl:if>
            <xsl:call-template name="AbbreviateNamePart">
              <xsl:with-param name="part" select="substring($part, 2)"/>
              <xsl:with-param name="previousWasChar" select="'true'" />
            </xsl:call-template>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:when>
      <xsl:otherwise>
        <!-- Add final dot. -->
        <xsl:text>.</xsl:text>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <!-- Format pages. -->
  <!--   The parameter 'pages' should be a string with page numbers. -->
  <xsl:template name="FormatPages">
    <xsl:param name="pages" />

    <xsl:if test="string-length($pages) > 0">
      <!-- Remove any possible spaces as well as 'p' and '.'. -->
      <xsl:variable name="pagesWithoutSpaces">
        <xsl:value-of select="translate($pages, '&#32;&#160;p.', '')"/>
      </xsl:variable>

      <!-- Check if this is about a single page, or a range of them. -->
      <xsl:choose>
        <xsl:when test="contains($pagesWithoutSpaces, '-')">
          <xsl:text>pp. </xsl:text>
        </xsl:when>
        <xsl:otherwise>
          <xsl:text>p. </xsl:text>
        </xsl:otherwise>
      </xsl:choose>
      <xsl:value-of select="$pagesWithoutSpaces"/>
    </xsl:if>

  </xsl:template>

  <!-- Format month. -->
  <xsl:template name="FormatMonth">
    <xsl:param name="month" />

    <!-- If month is a character string, grab the first three letters and
         convert them to lowercase characters to ease identifying the month. -->
    <xsl:variable name="shortmonth">
      <xsl:choose>
        <xsl:when test="string-length($month) >= 3">
          <xsl:variable name="temp" select="substring($month,1,3)"/>
          <xsl:value-of select="translate($temp,$upper,$lower)"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="$month"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <xsl:choose>
      <xsl:when test="(number($month) = 1) or string($shortmonth) = 'jan'">
        <xsl:text>Jan.</xsl:text>
      </xsl:when>
      <xsl:when test="(number($month) = 2) or string($shortmonth) = 'feb'">
        <xsl:text>Feb.</xsl:text>
      </xsl:when>
      <xsl:when test="(number($month) = 3) or string($shortmonth) = 'mar'">
        <xsl:text>Mar.</xsl:text>
      </xsl:when>
      <xsl:when test="(number($month) = 4) or string($shortmonth) = 'apr'">
        <xsl:text>Apr.</xsl:text>
      </xsl:when>
      <xsl:when test="(number($month) = 5) or string($shortmonth) = 'may'">
        <xsl:text>May</xsl:text>
      </xsl:when>
      <xsl:when test="(number($month) = 6) or string($shortmonth) = 'jun'">
        <xsl:text>Jun.</xsl:text>
      </xsl:when>
      <xsl:when test="(number($month) = 7) or string($shortmonth) = 'jul'">
        <xsl:text>Jul.</xsl:text>
      </xsl:when>
      <xsl:when test="(number($month) = 8) or string($shortmonth) = 'aug'">
        <xsl:text>Aug.</xsl:text>
      </xsl:when>
      <xsl:when test="(number($month) = 9) or string($shortmonth) = 'sep'">
        <xsl:text>Sep.</xsl:text>
      </xsl:when>
      <xsl:when test="(number($month) = 10) or string($shortmonth) = 'oct'">
        <xsl:text>Oct.</xsl:text>
      </xsl:when>
      <xsl:when test="(number($month) = 11) or string($shortmonth) = 'nov'">
        <xsl:text>Nov.</xsl:text>
      </xsl:when>
      <xsl:when test="(number($month) = 12) or string($shortmonth) = 'dec'">
        <xsl:text>Dec.</xsl:text>
      </xsl:when>
    </xsl:choose>
  </xsl:template>

  <!-- Format editors. -->
  <!--   The parameter 'list' should be either a b:NameList element or a b:Corporate element. -->
  <!--   The parameter 'maxEditors' should be a number. -->
  <xsl:template name="FormatEditors">
    <xsl:param name="list" />
    <xsl:param name="maxEditors" />

    <xsl:call-template name="FormatNameList">
      <xsl:with-param name="list" select="$list/b:NameList" />
      <xsl:with-param name="maxAuthors" select="$maxEditors" />
    </xsl:call-template>

    <xsl:choose>
      <xsl:when test="count($list/b:NameList/b:Person) = 1">
        <xsl:text>, Ed.</xsl:text>
      </xsl:when>
      <xsl:when test="count($list/b:NameList/b:Person) > 1">
        <xsl:text>, Eds.</xsl:text>
      </xsl:when>
    </xsl:choose>
  </xsl:template>

  <!-- Format ordinal. -->
  <!--   The parameter 'number' should be a number. -->
  <xsl:template name="FormatOrdinal">
    <xsl:param name="number" />

    <!-- Translate the number to digits only. -->
    <xsl:param name="digitsOnly">
      <xsl:value-of select="translate($number, 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ ,-.', '')"/>
    </xsl:param>

    <xsl:choose>
      <!-- If there are no digits, assume its one of the first ten ordinals written in full. -->
      <xsl:when test="string-length($digitsOnly) = 0">

        <!-- Translate the ordinal into lower cases only. -->
        <xsl:variable name="num">
          <xsl:value-of select="translate($number,$upper,$lower)"/>
        </xsl:variable>

        <xsl:choose>
          <xsl:when test="$num = 'first'">
            <xsl:text>1st</xsl:text>
          </xsl:when>
          <xsl:when test="$num = 'second'">
            <xsl:text>2nd</xsl:text>
          </xsl:when>
          <xsl:when test="$num = 'third'">
            <xsl:text>3rd</xsl:text>
          </xsl:when>
          <xsl:when test="$num = 'fourth'">
            <xsl:text>4th</xsl:text>
          </xsl:when>
          <xsl:when test="$num = 'fifth'">
            <xsl:text>5th</xsl:text>
          </xsl:when>
          <xsl:when test="$num = 'sixth'">
            <xsl:text>6th</xsl:text>
          </xsl:when>
          <xsl:when test="$num = 'seventh'">
            <xsl:text>7th</xsl:text>
          </xsl:when>
          <xsl:when test="$num = 'eighth'">
            <xsl:text>8th</xsl:text>
          </xsl:when>
          <xsl:when test="$num = 'nineth'">
            <xsl:text>9th</xsl:text>
          </xsl:when>
          <xsl:when test="$num = 'tenth'">
            <xsl:text>10th</xsl:text>
          </xsl:when>
          <!-- No idea on how to format this, just print it. -->
          <xsl:otherwise>
            <xsl:value-of select="$number"/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:when>
      <!-- So it is digit based, write it out with the appendix. -->
      <xsl:otherwise>
        <!-- Number. -->
        <xsl:value-of select="$digitsOnly"/>

        <!-- Ordinal appendix. -->
        <xsl:variable name="lastDigit">
          <xsl:value-of select="substring($digitsOnly, string-length($digitsOnly), 1)"/>
        </xsl:variable>

        <xsl:variable name="lastTwoDigits">
          <xsl:value-of select="substring($digitsOnly, string-length($digitsOnly) - 1, 2)"/>
        </xsl:variable>

        <xsl:choose>
          <xsl:when test="$lastTwoDigits = '11' or $lastTwoDigits = '12' or $lastTwoDigits = '13'">
            <xsl:text>th</xsl:text>
          </xsl:when>
          <xsl:when test="$lastDigit = '1'">
            <xsl:text>st</xsl:text>
          </xsl:when>
          <xsl:when test="$lastDigit = '2'">
            <xsl:text>nd</xsl:text>
          </xsl:when>
          <xsl:when test="$lastDigit = '3'">
            <xsl:text>rd</xsl:text>
          </xsl:when>
          <xsl:otherwise>
            <xsl:text>th</xsl:text>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:otherwise>
    </xsl:choose>

  </xsl:template>

  <!-- Translates a country name to a nationality. -->
  <!--   The parameter 'country' should be the name of the country. If no matching nationality is found, the country name will be returned. -->
  <xsl:template name="TranslateCountryToNationality">
    <xsl:param name="country" />

    <xsl:variable name="countrylow">
      <xsl:value-of select="translate($country, $upper, $lower)"/>
    </xsl:variable>

    <xsl:choose>
      <xsl:when test="$countrylow = 'albania'">
        <xsl:text>Albanian</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'australia'">
        <xsl:text>Australian</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'austria'">
        <xsl:text>Austrian</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'belgium'">
        <xsl:text>Belgian</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'bulgary'">
        <xsl:text>Bulgarian</xsl:text>
      </xsl:when>
      <xsl:when test="contains($countrylow,'china')">
        <xsl:text>Chinese</xsl:text>
      </xsl:when>
      <xsl:when test="contains($countrylow,'czech')">
        <xsl:text>Czech</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'danmark'">
        <xsl:text>Danish</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'england'">
        <xsl:text>English</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'estonia'">
        <xsl:text>Estonian</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'finland'">
        <xsl:text>Finnish</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'france'">
        <xsl:text>French</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'germany'">
        <xsl:text>German</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'greece'">
        <xsl:text>Greek</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'hungary'">
        <xsl:text>Hungarian</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'ireland'">
        <xsl:text>Irish</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'italy'">
        <xsl:text>Italian</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'japan'">
        <xsl:text>Japanese</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'lithuania'">
        <xsl:text>Lithuanian</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'korea'">
        <xsl:text>Korean</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'latvia'">
        <xsl:text>Latvian</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'norway'">
        <xsl:text>Norwegian</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'poland'">
        <xsl:text>Polish</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'portugal'">
        <xsl:text>Portugese</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'romania'">
        <xsl:text>Romanian</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'russia'">
        <xsl:text>Russian</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'scotland'">
        <xsl:text>Scottish</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'slovakia'">
        <xsl:text>Slovakian</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'spain'">
        <xsl:text>Spanish</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'switzerland'">
        <xsl:text>Swiss</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'taiwan'">
        <xsl:text>Taiwanese</xsl:text>
      </xsl:when>
      <xsl:when test="$countrylow = 'turkey'">
        <xsl:text>Turkish</xsl:text>
      </xsl:when>
      <xsl:when test="contains($countrylow, 'netherland')">
        <xsl:text>Dutch</xsl:text>
      </xsl:when>
      <xsl:when test="contains($countrylow, 'united') and contains($countrylow, 'kingdom')">
        <xsl:text>British</xsl:text>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="$country"/>
      </xsl:otherwise>
    </xsl:choose>

  </xsl:template>

  <!-- Formats a date. -->
  <!--   Returns a date as 'Abbriviated month' 'day', 'year'. -->
  <xsl:template name="FormatDate">
    <xsl:param name="day" />
    <xsl:param name="month" />
    <xsl:param name="year" />

    <xsl:if test="string-length($year) > 0">
      <xsl:if test="string-length($month) > 0">
        <xsl:call-template name="FormatMonth">
          <xsl:with-param name="month" select="$month" />
        </xsl:call-template>
        <xsl:text> </xsl:text>
        <xsl:if test="string-length($day) > 0">
          <xsl:value-of select="$day"/>
          <xsl:text>, </xsl:text>
        </xsl:if>
      </xsl:if>
      <xsl:value-of select="$year" />
    </xsl:if>

  </xsl:template>

  <!-- Format a URL. -->
  <!--   The parameter 'url' should be a string containing a valid URL. -->
  <!--   The parameter 'displayAsLink' should contain the value 'true' of 'false'
         and indicates if the URL should be displayed as a clickable link or not. 
         -->
  <xsl:template name="FormatURL">
    <xsl:param name="url" />
    <xsl:param name="displayAsLink" select="'true'" />

    <xsl:if test="string-length($url) > 0">
      <xsl:text>[Online]. </xsl:text>

      <xsl:if test="$displayAsLink = 'true'">
        <xsl:text>&lt;a href="</xsl:text>
        <xsl:value-of select="$url"/>
        <xsl:text>"&gt;</xsl:text>
      </xsl:if>

      <xsl:value-of select="$url"/>

      <xsl:if test="$displayAsLink = 'true'">
        <xsl:text>&lt;/a&gt;</xsl:text>
      </xsl:if>
    </xsl:if>

  </xsl:template>

  <!-- Replace part of a string with another string. -->
  <xsl:template name="ReplaceText">
    <xsl:param name="text"/>
    <xsl:param name="from"/>
    <xsl:param name="to"/>

    <xsl:choose>
      <xsl:when test="contains($text, $from)">
        <!-- Text before the replacement. -->
        <xsl:value-of select="substring-before($text, $from)"/>
        <!-- Replacement. -->
        <xsl:value-of select="$to"/>
        <!-- Text after the replacement (recursive). -->
        <xsl:call-template name="ReplaceText">
          <xsl:with-param name="text" select="substring-after($text, $from)"/>
          <xsl:with-param name="from" select="$from"/>
          <xsl:with-param name="to" select="$to"/>
        </xsl:call-template>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="$text"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>
  
  <!-- End of templates with shared functionality between the different types of references. -->

</xsl:stylesheet>


